package com.phonepe.sdk.javasdk.http;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.ExtractedResponse;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import org.apache.commons.lang3.ObjectUtils;

import java.nio.charset.Charset;
import java.util.List;

import static com.phonepe.sdk.javasdk.http.utils.HttpUtils.checkNotNull;


@Slf4j
public class PhonePeHttpCommand<T> {

    /* create and initialize this OkHttp client */
    private OkHttpClient client;
    /* response type class */
    private Class<T> responseType;
    /* response type reference class */
    private TypeReference<T> responseTypeReference;
    /* Builds the response in case of an exception in the http call */
    private Consumer<Exception, T> exceptionConsumer;
    /* Builds the response in case of nonSuccessful http call */
    private Consumer<ExtractedResponse, T> nonSuccessResponseConsumer;

    @Builder
    PhonePeHttpCommand(final OkHttpClient client,
                       final Class<T> responseType,
                       final Consumer<Exception, T> exceptionConsumer,
                       final Consumer<ExtractedResponse, T> nonSuccessResponseConsumer,
                       final TypeReference<T> responseTypeReference) {
        this.client = client;
        this.responseType = responseType;
        this.exceptionConsumer = exceptionConsumer;
        this.nonSuccessResponseConsumer = nonSuccessResponseConsumer;
        this.responseTypeReference = responseTypeReference;
    }

    <J> RequestBody getRequestBody(final PhonePeHttpRequest<J> httpRequest) throws JsonProcessingException {
        return RequestBody.create(MediaType.parse(HttpUtils.APPLICATION_JSON),
                                  MapperUtils.getMapper().writeValueAsBytes(httpRequest.getData()));
    }

    public void buildHeaders(Request.Builder requestBuilder, List<HttpHeaderPair> headers) {
        if (ObjectUtils.anyNotNull(headers)) {
            for (HttpHeaderPair httpHeader : headers) {
                requestBuilder.header(httpHeader.getName(),
                                      httpHeader.getValue());
            }
        }
    }

    public T doGet(final HttpUrl httpUrl,
                   final List<HttpHeaderPair> headers) throws PhonePeClientException {
        preconditions(false, httpUrl, null);
        Request.Builder requestBuilder = new Request.Builder()
                .url(httpUrl)
                .get();
        buildHeaders(requestBuilder,headers);
        return execute(requestBuilder.build());
    }

    public <J> T doPost(final HttpUrl httpUrl,
                        final List<HttpHeaderPair> headers,
                        final PhonePeHttpRequest<J> httpRequest) throws JsonProcessingException,PhonePeClientException {
        preconditions(true, httpUrl, httpRequest);
        Request.Builder requestBuilder = new Request.Builder()
                .url(httpUrl)
                .post(getRequestBody(httpRequest));
        buildHeaders(requestBuilder, headers);
        return execute(requestBuilder.build());
    }

    public T execute(Request request) throws PhonePeClientException {
        try {
            log.info("Service request: {}",
                     request);
            final Response response = client
                    .newCall(request)
                    .execute();

            /* OkHttpUtils.body ensures that the response is closed */
            final byte[] responseBody = HttpUtils.body(response);
            if (!response.isSuccessful()) {
                if (nonSuccessResponseConsumer != null) {
                    return nonSuccessResponseConsumer.consume(extract(response,
                                                                      responseBody));
                }
                log.error(String.format("Service call failed statusCode:%d request:%s response:%s",
                                        response.code(),
                                        request,
                                        new String(responseBody)));
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                                 "Service call failure: " + response.code());
            }
            if (ObjectUtils.anyNotNull(responseBody)) {
                if (responseTypeReference != null) {
                    return MapperUtils.getMapper().readValue(responseBody,
                                            responseTypeReference);
                }
                if (byte[].class.equals(responseType)) {
                    return (T) responseBody;
                }
                if (String.class.equals(responseType)) {
                    return (T) new String(responseBody,
                                          Charset.defaultCharset());
                }
                return MapperUtils.getMapper().readValue(responseBody,
                                        responseType);
            }
            return null;
        } catch (Exception e) {
            if (exceptionConsumer != null) {
                return exceptionConsumer.consume(e);
            }
            log.error(String.format("Error executing request:%s error:%s",
                                    request,
                                    e.getMessage()),
                      e);
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,
                                             e.getMessage(),
                                             e);
        }
    }

    public <J> void preconditions(boolean isPost, HttpUrl httpUrl, final PhonePeHttpRequest<J> httpRequest) throws PhonePeClientException {
        Preconditions.checkNotNull(client,
                                   "client cannot be null");
        Preconditions.checkNotNull(httpUrl,
                                   "httpUrl cannot be null");
        checkNotNull(responseType,
                     responseTypeReference,
                     "both responseType and responseTypeReference cannot be null");
        if(isPost)
            Preconditions.checkNotNull(httpRequest,
                                       "postData cannot be null");

    }

    private ExtractedResponse extract(final Response response,
                                      final byte[] responseBody) {
        return ExtractedResponse
                .builder()
                .body(responseBody)
                .code(response.code())
                .headers(response.headers())
                .message(response.message())
                .protocol(response.protocol())
                .handshake(response.handshake())
                .receivedResponseAtMillis(response.receivedResponseAtMillis())
                .sentRequestAtMillis(response.sentRequestAtMillis())
                .build();
    }
}

