package com.phonepe.sdk.javasdk.http.utils;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.phonepe.sdk.javasdk.config.models.DefaultConfig;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.transaction.init.models.enums.CallbackMode;
import com.phonepe.sdk.javasdk.transaction.init.models.enums.RedirectMode;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * Utils class containing helper methods for making Http calls.
 */
@Slf4j
@UtilityClass
public final class HttpUtils {

    public static final String APPLICATION_JSON = "application/json";
    public static final String CONTENT_TYPE = "Content-Type";
    public static final String CHECKSUM_HEADER = "X-VERIFY";
    public static final String LOCATION_HEADER = "location";
    public static final String PROVIDER_ID_HEADER = "X-PROVIDER-ID";
    public static final String REDIRECT_URL_HEADER = "X-REDIRECT-URL";
    public static final String REDIRECT_MODE_HEADER = "X-REDIRECT-MODE";
    public static final String CALLBACK_URL_HEADER = "X-CALLBACK-URL";
    public static final String CALL_MODE_HEADER = "X-CALL-MODE";
    public static final String V1_DEBIT_URL = "/v1/debit";
    public static final String V3_DEBIT_URL = "/v3/debit";
    public static final String JAVA_SDK_VERSION_HEADER = "x-java-sdk-version";
    public static final String DEFAULT_VERSION = "DEFAULT";
    public static final String VERSION = "version";

    public static byte[] body(final Response response) throws IOException {
        final ResponseBody body = response.body();
        if(ObjectUtils.anyNotNull(body))
            return body.bytes();
        return new byte[]{};
    }

    public static HttpHeaderPair getHeaderPair(final String name, final String value){
        return HttpHeaderPair.builder()
                             .name(name)
                             .value(value)
                             .build();
    }

    public static HttpHeaderPair getContentTypeHeaderPair(){
        return getHeaderPair(CONTENT_TYPE, APPLICATION_JSON);
    }

    public static <T1, T2> void checkNotNull(T1 t, T2 t2, String message) throws PhonePeClientException{
        if (t == null && t2 == null) {
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,message);
        }
    }

    public static List<HttpHeaderPair> getHttpHeaders(final String checksum,
                                                      final String providerId,
                                                      final String redirectURL,
                                                      final RedirectMode redirectMode,
                                                      final String callbackURL,
                                                      final CallbackMode callMode) {
        Preconditions.checkNotNull(checksum);
        List<HttpHeaderPair> headerPairs = getHttpHeaders(checksum);
        if(!Strings.isNullOrEmpty(providerId)){
            HttpHeaderPair providerIdHeader = getHeaderPair(PROVIDER_ID_HEADER, providerId);
            headerPairs.add(providerIdHeader);
        }
        if(!Strings.isNullOrEmpty(redirectURL)){
            HttpHeaderPair redirectURLHeader = getHeaderPair(REDIRECT_URL_HEADER, redirectURL);
            headerPairs.add(redirectURLHeader);
        }
        if(ObjectUtils.allNotNull(redirectMode)){
            HttpHeaderPair redirectModeHeader = getHeaderPair(REDIRECT_MODE_HEADER, redirectMode.getValue());
            headerPairs.add(redirectModeHeader);
        }
        if(!Strings.isNullOrEmpty(callbackURL)){
            HttpHeaderPair callbackURLHeader = getHeaderPair(CALLBACK_URL_HEADER, callbackURL);
            headerPairs.add(callbackURLHeader);
        }
        if(ObjectUtils.allNotNull(callMode)){
            HttpHeaderPair callbackModeHeader = getHeaderPair(CALL_MODE_HEADER, callMode.getValue());
            headerPairs.add(callbackModeHeader);
        }
        return headerPairs;
    }

    public static List<HttpHeaderPair> getHttpHeaders(final String checksum) {
        List<HttpHeaderPair> headerPairs = new ArrayList<>();
        HttpHeaderPair checksumHeader = getHeaderPair(CHECKSUM_HEADER, checksum);
        headerPairs.add(checksumHeader);
        HttpHeaderPair contentTypeHeaderPair = getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        return headerPairs;
    }

    public static boolean isHttpEndpointSecured(final Endpoint endpoint){
        return endpoint.getOverrideHost().contains("https");
    }

    public static HttpHeaderPair getJavaSdkVersionHeader(final DefaultConfig defaultConfig) throws IOException{
        String version = DEFAULT_VERSION;
        final Properties p = new Properties();
        final InputStream is = MethodHandles.lookup().lookupClass().getResourceAsStream(defaultConfig.getPomPropertiesFilePath());
        if (is != null) {
            p.load(is);
            version = p.getProperty(VERSION, DEFAULT_VERSION);
        }
        // fallback to using Java API
        if (version.equalsIgnoreCase(DEFAULT_VERSION)) {
            version = getSdkVersionFromAPI();
        }
        return getHeaderPair(JAVA_SDK_VERSION_HEADER, version);
    }

    private static String getSdkVersionFromAPI() {
        String apiVersion = DEFAULT_VERSION;
        Package aPackage = MethodHandles.lookup().lookupClass().getPackage();
        if (aPackage != null) {
            apiVersion = StringUtils.isNotEmpty(aPackage.getImplementationVersion()) ? aPackage.getImplementationVersion() : DEFAULT_VERSION;
            if (apiVersion.equalsIgnoreCase(DEFAULT_VERSION)) {
                apiVersion = StringUtils.isNotEmpty(aPackage.getSpecificationVersion()) ? aPackage.getSpecificationVersion() : DEFAULT_VERSION;
            }
        }
        return apiVersion;
    }
}

