package com.phonepe.sdk.javasdk.transaction.callback;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.transaction.models.Base64Request;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.utils.Base64Utils;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

@Slf4j
public class CallbackHandler {

    private TransactionStatusChecker transactionStatusChecker;
    private MerchantConfig merchantConfig;
    private ExecutorService executorService;
    private StatusResponseHandler defaultResponseHandler;

    @Builder
    public CallbackHandler(final TransactionStatusChecker transactionStatusChecker,
                           final MerchantConfig merchantConfig) {
        this.transactionStatusChecker = transactionStatusChecker;
        this.merchantConfig = merchantConfig;
        this.executorService = Executors.newCachedThreadPool();
        this.defaultResponseHandler = new DefaultStatusResponseHandler();
    }

    public void handleCallback(final String responseReceived,
                               final String checksum,
                               final long transactionAmount,
                               final StatusResponseHandler customResponseHandler) throws PhonePeClientException {
        Preconditions.checkNotNull(responseReceived);
        Preconditions.checkNotNull(checksum);
        try {
            final String responseBody = Base64Utils.decodeBase64(responseReceived);
            final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse = MapperUtils
                    .getMapper()
                    .readValue(responseBody,
                               new TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {
                               });
            final int keyIndex = KeyUtils.getKeyIndexFromChecksum(checksum);
            final TransactionStatusResponse callbackResponseData = transactionStatusResponse.getData();
            final String merchantId = callbackResponseData.getMerchantId();
            final Long callbackAmount = callbackResponseData.getAmount();
            final List<String> params = getParamsList(callbackResponseData);
            validateChecksum(responseReceived,
                             checksum,
                             params,
                             keyIndex);
            validateTransactionAmount(callbackAmount,
                                      transactionAmount);
            validateMerchantId(merchantId);
            final String transactionId = transactionStatusResponse
                    .getData()
                    .getTransactionId();
            this.executorService.submit(getRunnable(callbackAmount,
                                                    transactionId,
                                                    keyIndex,
                                                    ObjectUtils.allNotNull(customResponseHandler) ? customResponseHandler : defaultResponseHandler));
        } catch (PhonePeClientException e) {
            log.error("Error occurred while handling callback: {}",
                      e.getMessage());
            throw e;
        } catch (Exception ex) {
            log.error("Error occurred while handling callback: {}",
                      ex.getMessage());
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put("MESSAGE",
                          ex.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error occurred while handling callback: " +
                                             ex.getMessage(),
                                             objectMap,
                                             ex);
        }
    }

    private void validateMerchantId(final String merchantId) throws PhonePeClientException {
        if (this.merchantConfig
                .getMerchantId()
                .equals(merchantId)) {
            return;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Invalid merchantId in callback response");
    }

    private void validateChecksum(final String responseBody,
                                  final String checksum,
                                  final List<String> params,
                                  final int keyIndex) throws PhonePeClientException {
        final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),
                                                          keyIndex);
        final Base64Request base64Request = Base64Request
                .builder()
                .request(responseBody)
                .build();
        final PhonePeHttpRequest<Base64Request> phonePeHttpRequest = PhonePeHttpRequest.<Base64Request>builder()
                .data(base64Request)
                .build();
        final String generatedChecksum = this.transactionStatusChecker
                .getChecksumGenerator()
                .getChecksumValue(phonePeHttpRequest,
                                  "",
                                  params,
                                  apiKey,
                                  keyIndex);
        if (checksum.equalsIgnoreCase(generatedChecksum)) {
            return;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Invalid checksum in callback response");
    }

    private void validateTransactionAmount(final Long callbackAmount,
                                           final Long statusResponseAmount) throws PhonePeClientException {
        if (callbackAmount.equals(statusResponseAmount)) {
            return;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Transaction amount mismatch in callback response and status check response " +
                                         "or in provided amount");
    }

    private List<String> getParamsList(final TransactionStatusResponse transactionStatusResponse) {
        final List<String> params = new ArrayList<>();
        params.add(transactionStatusResponse.getMerchantId());
        params.add(transactionStatusResponse.getTransactionId());
        params.add(transactionStatusResponse
                           .getAmount()
                           .toString());
        return params;
    }

    private Runnable getRunnable(final long callbackAmount,
                                 final String transactionId,
                                 final int keyIndex,
                                 final StatusResponseHandler statusResponseHandler) {

        return new Runnable() {
            @Override
            public void run() {
                try{
                    StatusResponse statusResponse = transactionStatusChecker.checkTransactionStatus(transactionId,
                                                                                                    keyIndex);
                    validateTransactionAmount(callbackAmount,
                                              statusResponse.getAmount());
                    statusResponseHandler.processStatusResponse(statusResponse);

                }  catch (final Exception ex){
                    log.error("Error occurred while handling callback: Issue while making status call and processing response: {}", ex.getMessage());
                }
            }
        };
    }
}