package com.phonepe.sdk.javasdk.transaction.status;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.google.common.base.Enums;
import com.google.common.base.Predicate;
import com.phonepe.sdk.javasdk.config.URLCreator;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.config.models.enums.ConfigType;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionCommand;
import com.phonepe.sdk.javasdk.transaction.models.enums.TransactionStatusCode;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.enums.PaymentState;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import com.phonepe.sdk.javasdk.utils.RetryUtils;
import lombok.Builder;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import static com.phonepe.sdk.javasdk.transaction.models.enums.TransactionStatusCode.INTERNAL_SERVER_ERROR;
import static com.phonepe.sdk.javasdk.transaction.models.enums.TransactionStatusCode.PAYMENT_PENDING;

@Slf4j
public class TransactionStatusChecker {

  private TransactionCommand transactionCommand;
  private StatusConfig statusConfig;
  private MerchantConfig merchantConfig;
  @Getter private ChecksumGenerator checksumGenerator;
  private PaymentStateCreator paymentStateCreator;
  private URLCreator urlCreator;

  @Builder
  public TransactionStatusChecker(
      final TransactionCommand transactionCommand,
      final MerchantConfig merchantConfig,
      final StatusConfig statusConfig,
      final ChecksumGenerator checksumGenerator,
      final PaymentStateCreator paymentStateCreator,
      final URLCreator urlCreator) {
    this.transactionCommand = transactionCommand;
    this.merchantConfig = merchantConfig;
    this.statusConfig = statusConfig;
    this.checksumGenerator = checksumGenerator;
    this.paymentStateCreator = paymentStateCreator;
    this.urlCreator = urlCreator;
  }

  public StatusResponse checkTransactionStatus(final String transactionId, final int keyIndex)
      throws PhonePeClientException {
    try {
      final HttpUrl httpUrl = getHttpUrl(transactionId);
      final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(), keyIndex);
      final List<String> params = getParamsList(transactionId);
      final String checksumHeaderValue =
          this.checksumGenerator.getChecksumValue(
              null, httpUrl.encodedPath(), params, apiKey, keyIndex);
      List<HttpHeaderPair> httpHeaders = HttpUtils.getHttpHeaders(checksumHeaderValue);
      Callable<PhonePeHttpResponse<TransactionStatusResponse>> transactionStatusCallable =
          getCallable(httpUrl, httpHeaders);
      Predicate<PhonePeHttpResponse<TransactionStatusResponse>> phonePeHttpResponsePredicate =
          getRetryPredicate();
      Retryer<PhonePeHttpResponse<TransactionStatusResponse>> retryer =
          RetryUtils.getRetryer(statusConfig.getRetryConfig(), phonePeHttpResponsePredicate);
      PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse =
          retryer.call(transactionStatusCallable);
      return buildStatusResponse(phonePeHttpResponse);
    } catch (RetryException ex) {
      PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse =
          (PhonePeHttpResponse<TransactionStatusResponse>) ex.getLastFailedAttempt().getResult();
      return buildStatusResponse(phonePeHttpResponse);
    } catch (Exception ex) {
      log.error("Exception occurred while performing transaction status check");
      Map<String, Object> objectMap = new HashMap<>();
      objectMap.put("MESSAGE", ex.getMessage());
      throw new PhonePeClientException(
          PhonePeClientException.ErrorCode.EXECUTION_ERROR,
          "Error executing check transaction status" + ": " + ex.getMessage(),
          objectMap,
          ex);
    }
  }

  private List<String> getParamsList(final String transactionId) {
    List<String> paramsList = new ArrayList<>();
    paramsList.add(this.merchantConfig.getMerchantId());
    paramsList.add(transactionId);
    return paramsList;
  }

  private HttpUrl getHttpUrl(final String transactionId) {
    return this.urlCreator.getHttpUrl(
            ConfigType.STATUS,
        this.statusConfig.getOverrideHost(),
        this.statusConfig.getApiVersion().getValue(),
        this.merchantConfig.getMerchantId(),
        transactionId);
  }

  private Callable<PhonePeHttpResponse<TransactionStatusResponse>> getCallable(
      final HttpUrl httpUrl, final List<HttpHeaderPair> headerPairs) {
    return new Callable<PhonePeHttpResponse<TransactionStatusResponse>>() {
      @Override
      public PhonePeHttpResponse<TransactionStatusResponse> call() throws Exception {
        return transactionCommand.getTransactionStatus(headerPairs, httpUrl);
      }
    };
  }

  private Predicate<PhonePeHttpResponse<TransactionStatusResponse>> getRetryPredicate() {
    return new Predicate<PhonePeHttpResponse<TransactionStatusResponse>>() {
      @Override
      public boolean apply(
          PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse) {
        final TransactionStatusCode code = Enums.getIfPresent(TransactionStatusCode.class, transactionStatusResponse.getCode()).or(TransactionStatusCode.UNKNOWN);
        return PAYMENT_PENDING == code
            || INTERNAL_SERVER_ERROR == code;
      }
    };
  }

  private StatusResponse buildStatusResponse(
      final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse) throws PhonePeClientException{
    return StatusResponse.builder()
        .transactionId(transactionStatusResponse.getData().getTransactionId())
        .paymentState(getPaymentState(transactionStatusResponse.getCode()))
        .providerReferenceId(transactionStatusResponse.getData().getProviderReferenceId())
        .amount(transactionStatusResponse.getData().getAmount())
        .paidAmount(transactionStatusResponse.getData().getPaidAmount())
        .paymentModes(transactionStatusResponse.getData().getPaymentModes())
        .build();
  }

  private PaymentState getPaymentState(final String statusCode) throws PhonePeClientException{
    final TransactionStatusCode code = Enums.getIfPresent(TransactionStatusCode.class, statusCode).or(TransactionStatusCode.UNKNOWN);
    return code.accept(this.paymentStateCreator);
  }
}
