package com.phonepe.sdk.javasdk.config;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.exception.PhonePeConfigurationException;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.Builder;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

public class ConfigurationFactory<T> {

    private final Class<T> clazz;
    private final ObjectMapper objectMapper;
    private final Validator validator;
    private final JsonFactory parserFactory;
    private static final String DEFAULT_SDK_CONFIG_FILE = "phonepe_java_sdk_properties.yml";

    @Builder
    public ConfigurationFactory(Class<T> clazz) {
        this.clazz = clazz;
        this.parserFactory = new YAMLFactory();
        this.validator = Validation
                .buildDefaultValidatorFactory()
                .getValidator();
        this.objectMapper = MapperUtils.getMapper();
    }

    public T buildConfig() throws PhonePeConfigurationException {
        return buildConfig(DEFAULT_SDK_CONFIG_FILE);
    }

    public T buildConfig(final String fileName) throws PhonePeConfigurationException {
        Preconditions.checkNotNull(fileName);
        File file;
        try {
            file = new File(fileName);
            InputStream inputStream = getClass()
                    .getClassLoader()
                    .getResourceAsStream(fileName);
            final JsonNode node = objectMapper.readTree(createParser(inputStream));
            final String filename = file.toString();
            return build(node,
                         filename);
        } catch (Exception exception) {
            throw new PhonePeConfigurationException(fileName,
                                                    exception.getMessage());
        }
    }

    protected JsonParser createParser(final InputStream inputStream) throws IOException {
        return parserFactory.createParser(inputStream);
    }

    private T build(final JsonNode node,
                    final String filename) throws IOException, PhonePeConfigurationException {
        final T config = objectMapper.readValue(new TreeTraversingParser(node),
                                                clazz);
        validate(filename,
                 config);
        return config;
    }

    private void validate(final String file,
                          T config) throws PhonePeConfigurationException {
        if (validator != null) {
            final Set<ConstraintViolation<T>> violations = validator.validate(config);
            if (!violations.isEmpty()) {
                throw new PhonePeConfigurationException(file,
                                                        violations);
            }
        }
    }
}