package com.phonepe.sdk.javasdk.transaction.client;


import com.fasterxml.jackson.core.type.TypeReference;
import com.phonepe.sdk.javasdk.config.models.DefaultConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.Consumer;
import com.phonepe.sdk.javasdk.http.PhonePeHttpCommand;
import com.phonepe.sdk.javasdk.http.models.ExtractedResponse;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class TransactionCommand {

    private final OkHttpClient client;
    private final DefaultConfig defaultConfig;
    private final List<HttpHeaderPair> metaHeaders;
    private static final String INITIATE_TRANSACTION = "initTransaction";
    private static final String TRANSACTION_STATUS = "transactionStatus";


    public TransactionCommand(final OkHttpClient okHttpClient,
                              final DefaultConfig defaultConfig) throws IOException{
        this.client = okHttpClient;
        this.defaultConfig= defaultConfig;
        this.metaHeaders = getMetaHeaders();
    }

    public <T> InitResponse initTransaction(final PhonePeHttpRequest<T> initRequest,
                                            List<HttpHeaderPair> httpHeaders,
                                            final HttpUrl httpUrl) throws PhonePeClientException {
        try {

            Consumer<ExtractedResponse, PhonePeHttpResponse<InitResponse>> initResponseConsumer =
                    handleNonSuccessServiceCall(log,
                                                INITIATE_TRANSACTION);
            httpHeaders.addAll(this.metaHeaders);
            PhonePeHttpResponse<InitResponse> phonePeHttpResponse =
                    PhonePeHttpCommand.<PhonePeHttpResponse<InitResponse>>builder()
                            .client(this.client)
                            .responseTypeReference(new TypeReference<PhonePeHttpResponse<InitResponse>>() {
                            })
                            .nonSuccessResponseConsumer(initResponseConsumer)
                            .build()
                            .doPost(httpUrl, httpHeaders, initRequest);

            return phonePeHttpResponse.getData();
        } catch (PhonePeClientException e) {
            log.error("Error while making call to PhonePe init transaction API: {}", e.getMessage());
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API: {}",
                      e.getMessage());
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put("MESSAGE",
                          e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error executing http client for initiating transaction: " +
                                             e.getMessage(),
                                             objectMap,
                                             e);
        }
    }


    public PhonePeHttpResponse<TransactionStatusResponse> getTransactionStatus(List<HttpHeaderPair> httpHeaders,
                                                                               final HttpUrl httpUrl) throws PhonePeClientException {
        try {
            Consumer<ExtractedResponse, PhonePeHttpResponse<TransactionStatusResponse>> transactionResponseConsumer =
                    handleNonSuccessServiceCall(log,
                                                TRANSACTION_STATUS);
            httpHeaders.addAll(this.metaHeaders);
            return PhonePeHttpCommand.<PhonePeHttpResponse<TransactionStatusResponse>>builder()
                    .client(this.client)
                    .responseTypeReference(new TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {
                    })
                    .nonSuccessResponseConsumer(transactionResponseConsumer)
                    .build()
                    .doGet(httpUrl, httpHeaders);
        } catch (PhonePeClientException e) {
            log.error("Error while making call to PhonePe transaction status API ");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe transaction status API ",
                      e);
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put("MESSAGE",
                          e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error executing http client for transaction status: " +
                                             e.getMessage(),
                                             objectMap,
                                             e);
        }
    }

    public static <T> Consumer<ExtractedResponse, T> handleNonSuccessServiceCall(final Logger log,
                                                                                 final String serviceName) {
        return new Consumer<ExtractedResponse, T>() {
            @Override
            public T consume(ExtractedResponse extractedResponse) throws PhonePeClientException {
                final int code = extractedResponse.getCode();
                final byte[] body = extractedResponse.getBody();
                final String bodyString = new String(body);
                log.error("Error while calling {} service code: {}, response:{}",
                          serviceName,
                          code,
                          bodyString);
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_SERVER_ERROR,
                                                 serviceName + " http call returned code " + code);
            }
        };
    }

    private List<HttpHeaderPair> getMetaHeaders() throws IOException {
        List<HttpHeaderPair> metaHeadersList = new ArrayList<>();
        metaHeadersList.add(HttpUtils.getJavaSdkVersionHeader(this.defaultConfig));
        return metaHeadersList;
    }

}
