(ns ito.core
  (:require [riemann.client :as r])
  (:refer-clojure :exclude [time]))

(def client (atom nil))
(def service-prefix (atom nil))

(defn init! [prefix config]
  (reset! service-prefix prefix)
  (reset! client (r/tcp-client config)))

(defn- make-service [ident type]
  (str @service-prefix \space type \space ident))

(defn- record [service value extras]
  (when @client
    (r/send-event @client
                  (merge {:service service
                          :metric value
                          :ttl 15}
                         extras))))

(defn counter
  ([ident value] (counter ident value {:state "ok"}))
  ([ident value extras]
   (record (make-service ident "counters") value extras)))

(defn time [ident start-time]
  (let [duration-seconds (/ (- (System/currentTimeMillis) start-time) 1000)]
    (record (make-service ident "timers")
            duration-seconds
            {:state "ok"})
    (float duration-seconds)))

(defn error [app ex]
  (counter (str app " errors") 1 {:state "down"
                                  :tags ["exception"]
                                  :description (.getMessage ex)}))
