(ns schema.extensions.derived
  (:require [schema.core :refer [subschema-walker explain start-walker walker] :as s]
            [schema.coerce :as coerce]
            #?(:clj [schema.macros :as sm])
            [schema.extensions.util :refer [field-update field-meta]])
  #?(:cljs (:require-macros [schema.macros :as sm])))


;;; Derived
;;; schema elements derived on other schema elements at the same level

(defn derived
  "Calculate a field's value based on its parent object. Parameters are parent object and key name."
  [schema calculator]
  (field-update schema {:_derived {:calculator calculator}}))

(defn defaulting
  "Allow a field to be a default. Is not applied if a value exists. Takes a function or a value."
  [schema default]
  (derived schema (fn [a k] (cond (some? (a k)) (a k)
                                  (fn? default) (default)
                                  :otherwise    default))))

(defn massaging
  "Calculate a fields value based on its current value"
  [schema massager]
  (derived schema (fn [a k] (massager (get a k)))))

(defn auto-defaulting
  [schema]
  (defaulting schema (empty schema)))

(defn optional [schema]
  (-> schema
      s/maybe
      (defaulting nil)))

;; Helpers

(defn derived? [schema]
  (-> schema field-meta :_derived some?))

(defn- derived-calculator [schema]
  (-> schema field-meta :_derived :calculator))

(defn derive-coercion [schema]
  (let [kvs (and (map? schema)
                 (filter (fn [[k v]] (derived? v)) schema))]
    (when (and kvs (not (empty? kvs)))
          (fn [data]
            (if-not (map? data) data
              (->> kvs
                   (map
                    (fn [[k ds]]
                      (try [k (s/validate ds ((derived-calculator ds) data k))]
                           (catch #?(:clj Exception
                                     :cljs js/Error) e
                          [k (sm/validation-error ds data 'deriving-exception e)]))))
                   (apply concat)
                   (apply assoc data)))))))

(defn- ignore-derived-fields
  "We don't want to validate derived fields because they'll get validated by
  derive-coercion once they've been derived. Before they've been derived, of course
  they're going to be invalid."
  [walk s]
  (fn [data]
    (if (and (= (type s) schema.core.MapEntry)
             (derived? (:val-schema s)))
      (when-not (= data :schema.core/missing)
        (if (coll? (second data))
          (walk data)
          data))
      (walk data))))

(defn- deriver [schema]
  (start-walker
   (fn [s]
     (let [walk (ignore-derived-fields (walker s) s)
           dc (or (derive-coercion s) identity)
           c (or (coerce/json-coercion-matcher s) identity)]
       (fn [x]
         ;; derive-coercion runs after walk here because we need to coerce json
         ;; before we allow derivation functions to run.
         ;; see https://github.com/Rafflecopter/raflui/issues/172#issuecomment-221173615
         ;; validation of the derived value happens in derive-coercion
         (-> x c walk dc))))
   schema))

;; Deriving walker

(defn derive-schema [schema data]
  "Walk a schema, deriving any keys that are of the Derived type"
  ((deriver schema) data))
