(ns schema.extensions.partial-validator
  (:require [schema.core :as s]
            [schema.spec.core :as spec]
            [schema.utils :as sutils]
            [schema.coerce :as coerce]
            #?(:clj [schema.macros :as sm])
            [schema.extensions.util :refer [field-meta]])
  #?(:cljs (:require-macros [schema.macros :as sm])))


;;;; NOTE: we might want to replace this stuff with Schema's new "completion":
;;;;       https://github.com/plumatic/schema/blob/master/src/clj/schema/experimental/complete.clj


(defn not-modifiable? [s]
  (-> s field-meta :modifiable false?))

(defn- make-partial-walker
  "Returns a function that, when invoked, will run a custom checking procedure on
  schema. We only want to validate a subset of data, so we ignore missing required keys."
  [f]
  (fn [input-schema]
    (spec/run-checker
     (fn [schema params]
       (let [checker (spec/checker (s/spec schema) params)
             unjson (or (coerce/json-coercion-matcher schema) identity)]
         (fn [data]
           (let [data (->> data unjson (f schema) checker)] ; Run checker to generate validation errors
             (if-not (sutils/error? data)
               ;; If no errors, pass through to provided validator
               (f schema data)
               ;; If a non-map error, then barf
               (if-not (map? (sutils/error-val data)) data
                       ;; If map of errors, remove the required-key errors we don't care about
                       (let [data (into {} (for [[k v] (sutils/error-val data)]
                                             (when-not (= v 'missing-required-key)
                                               [k v])))]
                         (if (empty? data)
                           data                ; If no errors left, carry on
                           (sutils/error data) ; Otherwise they're real errors, so barf
                           ))))))))
     true
     input-schema)))

;; Basic partial validator

(def partial-walker (make-partial-walker (fn [_ x] x)))

(defn partial-check [schema data]
  (-> schema
      partial-walker
      (#(% data))
      sutils/error-val))

(defn partial-validate
  "Throw an exception if value does not partially satisfy schema; otherwise, return value."
  [schema value]
  (when-let [error (partial-check schema value)]
    (sm/error! (sutils/format* "Value does not match schema: %s" (pr-str error))
                   {:schema schema :value value :error error}))
  value)

;; Not Modifiable check partial validator

(def not-modified-partial-walker
  (make-partial-walker
   (fn [schema x]
     (if (not-modifiable? schema)
       (sm/validation-error schema x 'modifiable)
       x))))

(defn not-modified-partial-check [schema data]
  (-> schema
      not-modified-partial-walker
      (#(% data))
      sutils/error-val))

(defn not-modified-partial-validate
  "Throw an exception if value does not satisfy schema; otherwise, return value."
  [schema value]
  (when-let [error (not-modified-partial-check schema value)]
    (sm/error! (sutils/format* "Value does not match schema: %s" (pr-str error))
                   {:schema schema :value value :error error}))
  value)
