(ns flowmaps.core
  (:refer-clojure :exclude [abs update-vals update-keys])
  (:require
    [clojure.set             :as cset]
    [flowmaps.utility        :as ut]
    [flowmaps.web            :as web]
    [flowmaps.db             :as db]
    [flowmaps.rest           :as rest]
    [debux.core              :as dx]
    [clojure.walk            :as walk]
    [clojure.edn             :as edn]
    [clojure.string          :as cstr]
    [websocket-layer.core    :as wl] ;; temp until refactor
    [clojure.spec.alpha      :as s]
    [chime.core              :as chime]
    [clojure.spec.test.alpha :as stest]
    [flowmaps.examples.simple-flows :as fex]
    [clojure.core.async      :as async])
  (:import
    [java.lang.management ManagementFactory]
    java.time.Instant
    java.time.LocalTime
    java.time.LocalDate
    [java.time            LocalTime Duration Instant ZonedDateTime ZoneId Period DayOfWeek]
    [java.text            SimpleDateFormat]
    [java.util            Date TimeZone Calendar]
    java.time.format.DateTimeFormatter))

(defmacro flow>
  [& forms]
  (let [step-count (atom 0)
        labels     (doall (map (fn [_] (do (swap! step-count inc) (keyword (str "step" @step-count)))) forms))
        conns      (vec (map vector labels (rest labels)))
        components (vec (mapcat vector labels forms))]
    `{:components ~(apply hash-map components) :connections ~conns}))




(s/def ::fn? (s/and ifn? #(not (map? %))))

(s/def ::components
  (s/or :function          ::fn?
        :map-with-function (s/and map? (s/keys :req-un [::fn?]))
        :static-value      #(and (not (fn? %)) (not (map? (:fn %))))))

(s/def ::connection-pair
  (s/and vector?
         (s/coll-of (s/or :keyword keyword?
                          :any     any?)
                    :count 2)))

(s/def ::connections (s/and vector? (s/and (s/coll-of ::connection-pair) (complement empty?))))

(s/def ::network-map (s/keys :req-un [::connections ::components]))

(s/def ::opts-map (s/keys :opt-un [::debug? ::debux?]))

(s/def ::done-ch #(or (ut/chan? %) (and (seqable? %) (ut/chan? (first %))) (instance? clojure.lang.Atom %)))

(s/def ::subflow-map map?)







(defn find-keys-with-path
  [paths-map path-segment]
  (let [segment-length (count path-segment)]
    (reduce (fn [acc [k v]]
              (if (some (fn [path]
                          (some #(= path-segment (subvec path % (+ % segment-length)))
                                (range 0 (+ 1 (- (count path) segment-length)))))
                        v)
                (conj acc k)
                acc))
      []
      paths-map)))

(defn build-all-paths
  [node connections visited]
  (let [next-nodes (map second (filter #(= (first %) node) connections))]
    (if (node (set visited))
      [[[node :loop-to] (first visited)]]
      (if (empty? next-nodes)
        [[node]]
        (for [next next-nodes path (build-all-paths next connections (cons node visited))] (cons node path))))))

(defn build-paths-from-all-nodes [connections] (mapcat #(build-all-paths % connections []) (set (map first connections))))

(defn path-map [connections] (into {} (for [v (build-paths-from-all-nodes connections)] {(hash v) (list (vec (flatten v)))})))

(defn- end-chain-msg
  [flow-id fp ff result the-path debug?] ;; TODO better calc of pathfinding to derive natural
  (try (let [fp-vals          (vec (for [f the-path] [f (get-in @db/results-atom [flow-id f])]))
             res-paths        (distinct (get @db/resolved-paths flow-id))
             chains           (count res-paths)
             last-in-chain?   (or (some #(= the-path %) res-paths) (= :error result) (= :timeout result))
             chains-run       (try (+ (count (distinct @db/chains-completed)) 1) (catch Exception _ 1))
             all-chains-done? (= chains chains-run)]
         (when debug? (ut/ppln [the-path :chain-done! ff :done-hop-or-empty :res-paths res-paths]))
         (when last-in-chain?
           (do (when debug?
                 (ut/ppln [chains-run :chain-done :done-hop ff :steps (count fp-vals) :final-result result :value-path fp-vals
                           chains-run :of chains :flow-chains-completed]))
               (swap! db/results-atom assoc-in [flow-id :done] result)
               (when (nil? (get-in @db/tracker [flow-id ff :end]))
                 (swap! db/tracker assoc-in [flow-id ff :end] (System/currentTimeMillis))) ;; stop all
               (when (nil? (get-in @db/status [flow-id :end])) ;; should this exist at all?
                 (swap! db/status assoc-in [flow-id :end] (System/currentTimeMillis)))
               (swap! db/chains-completed conj the-path))
           (when (and all-chains-done? last-in-chain?)
             (when debug?
               (ut/ppln [:all-chains-done! :resolved-paths (vec (distinct (get @db/resolved-paths flow-id))) ff :fp fp
                         :results-atom (get @db/results-atom flow-id)])))))
       (catch Exception _ nil))) ;; TODO weird bug when close-on-done? is true and we're using an atom as output channel

(defn close-channels!
  [flow-id & [hint]]
  (let [still-open?    (true? (some true? (for [c (vals (get @db/channels-atom flow-id))] (ut/chan? c))))
        all-blocks-in? (true? (every? some? (vals (get @db/results-atom flow-id))))]
    (when still-open? ;; if the channels are already closed, do nothing, we good.
      (swap! db/trigger-guard conj flow-id)
      (when (not all-blocks-in?) (Thread/sleep 5000)) ;; if not all blocks have data, wait a
      (doseq [[k c] (get @db/channels-atom flow-id)]
        (try #_{:clj-kondo/ignore [:redundant-do]}
             (do ;(ut/ppln [:closing-channel k c])
               (try (when (not (nil? c)) (do (swap! db/channels-atom ut/dissoc-in [flow-id k]) (async/close! c)))
                    (catch Throwable e
                      (do (swap! db/channels-atom assoc-in [flow-id k] c) (ut/ppln [:error-closing-channels-inner e k c])))))
             (catch Throwable e
               (do (swap! db/channels-atom assoc-in [flow-id k] c) ;; if failed, keep key val
                   (ut/ppln [:error-closing-channels-outer e k c])))))
      (swap! db/trigger-guard disj flow-id))))

(defn close-channel!
  [flow-id channel-id]
  (try (let [ch (get-in @db/channels-atom [flow-id channel-id])]
         (try (when (not (nil? ch)) (do (swap! db/channels-atom ut/dissoc-in [flow-id channel-id]) (async/close! ch)))
              (catch Throwable e (ut/ppln [:error-closing-channel-inner e flow-id channel-id]))) ;; close
         (swap! db/channels-atom ut/dissoc-in [flow-id channel-id]))
       (catch Throwable e (ut/ppln [:error-closing-channel-outer e flow-id channel-id]))))

(declare flow) ;; so we can call it in (process ..) even though it's still undefined at this point

(defn- process
  [flow-id base-flow-id connections input-chs output-chs ff fp block-function opts-map done-ch components]
  (let [;web-push? (true? @web/websocket?)
        {:keys [debug? close-on-done?]} opts-map
        timeout                         (get opts-map :timeout 120000) ;; wait for 2 mins max
        allow-repeats?                  (get opts-map :allow-repeats?)
        output-chs                      (remove nil? output-chs) ;; 2/23/24 ?
        channel-results                 (atom (into {} (map #(vector % nil) input-chs)))
        block-fn-raw?                   (fn? block-function)
        is-subflow?                     (s/valid? ::network-map block-function)
        inputs                          (get block-function :inputs [])
        block-is-port?                  (and (not block-fn-raw?) (get block-function :fn) (ut/namespaced? ff))]
    (try
      (async/go-loop [pending-chs input-chs]
        (when (seq pending-chs)
          (let [;_ (ut/ppln [:pc ff pending-chs])
                [data ch]              (async/alts! pending-chs)
                data-from              (get data :sender)
                data-val               (get data :value :done)
                repeat-input?          (and (= data-val (get-in @db/last-input [flow-id ff])) (not allow-repeats?))
                _ (swap! db/last-input assoc-in [flow-id ff] data-val)
                next-hops              (map second (filter #(= ff (first %)) connections))
                triples                [data-from ff (first next-hops)]
                poss-path-keys         (find-keys-with-path fp triples)
                is-accumulating-ports? (true? (get data-val :port-in?))
                start                  (System/currentTimeMillis) ;; non wait
                the-path               (try (apply max-key count (apply concat (vals (select-keys fp poss-path-keys))))
                                            (catch Exception _ [:error-path-issue!]))
                valid-output-channels? (or (ut/chan? done-ch) ;; single output channel
                                           (try               ;; or several
                                             (ut/chan? (first done-ch))
                                             (catch Exception _ false)))
                done-channel-push?     (and (= ff :done) valid-output-channels?)]
            (when (or (= data-val :timeout) (and (= ff :done) (not (ut/chan? data-val)))) ;; process any :done
                                                                                          ;; stuff, debug or not.
                                                                                          ;; ends this
                                                                                          ;; flow-chain-path
              (let [done-channel? valid-output-channels? ;(ut/chan? done-ch)
                    multi-ch?     (seqable? done-ch)
                    done-atom?    (instance? clojure.lang.Atom done-ch)]
                (cond done-atom?                    (when (not (= data-val :done)) (reset! done-ch data-val))                   ;;  TODO,
                                                                                                                                ;;  weird
                      (and done-channel? multi-ch?) (doseq [d done-ch] (async/>! d data-val))
                      done-channel?                 (async/>! done-ch data-val)
                      :else                         nil)))
            (when is-accumulating-ports?
              (swap! db/port-accumulator assoc-in [flow-id ff] (merge (get-in @db/port-accumulator [flow-id ff] {}) data-val))) ;; ff
                                                                                                                                ;; is
                                                                                                                                ;; the
            (swap! channel-results assoc ch data-val)
            (swap! db/channel-history update
              flow-id
              conj
              (let [v (if (and (map? data-val) (get data-val :port-in?)) (first (vals (dissoc data-val :port-in?))) data-val)]
                {;:path the-path ;; PRE RECUR HISTORY PUSH.
                 :type         :channel
                 :run-id       (get opts-map :run-id)
                 :channel      [data-from ff]
                 :base-flow-id (str base-flow-id)
                 :dest         ff
                 :start        start
                 :end          (System/currentTimeMillis)
                 :value        (ut/limited v flow-id)
                 :data-type    (ut/data-typer v)}))
            (when (or (= data-val :timeout) (and (= ff :done) (not (nil? data-val)) (not (ut/chan? data-val)) close-on-done?))
              (future (close-channels! flow-id 2)))
            (if (or (and (not (ut/namespaced? ff)) (not (ut/namespaced? data-from)))
                    (not (some nil?
                               (vals ;; dont wait for condi values, let them be a suprise?
                                 (ut/remove-keys @channel-results (vals (get-in @db/condi-channels-atom [flow-id])))))))
              (if (or (get data-val :error) (get data-val :timeout) (= data-val :done)) ;; receiving
                                                                                        ;; :done/error
                                                                                        ;; keyword
                                                                                        ;; as a
                                                                                        ;; value is
                                                                                        ;; a
                (do (when close-on-done? (future (close-channels! flow-id 1))) ;; non blocking
                    (when (or (get data-val :error) (= data-val :timeout)) ;; if ending due to
                      (swap! db/results-atom assoc-in [flow-id :done] [:error! ff data-val])
                      (swap! db/tracker assoc-in [flow-id ff :end] (System/currentTimeMillis)) ;; stop
                      (swap! db/status assoc-in [flow-id :end] (System/currentTimeMillis))) ;; force
                    (end-chain-msg flow-id fp ff nil the-path debug?))
                (let [;block-fn-raw? (fn? block-function)
                      resolved-inputs (walk/postwalk-replace (get-in @db/port-accumulator [flow-id ff]) inputs)
                      fully-resolved? (= 0 (count (cset/intersection (set resolved-inputs) (set inputs))))
                      pre-in (cond (and fully-resolved?
                                        (not is-subflow?) ;; subflows namespaced keys are not
                                        is-accumulating-ports?)
                                     resolved-inputs
                                   :else data-val)
                      pre-when? (try (if (and fully-resolved? is-accumulating-ports?)
                                       (apply (get block-function :pre-when? (fn [_] true)) pre-in) ;; resolved
                                       ((get block-function :pre-when? (fn [_] true)) pre-in))
                                     (catch Exception _ true)) ;; hmm, maybe false instead?
                      subflow-overrides (when is-subflow?
                                          (into {}
                                                (for [[k v] (dissoc (get-in @db/port-accumulator [flow-id ff]) ;data-val
                                                              :port-in?)]
                                                  {(keyword (last (cstr/split (str (ut/unkeyword k)) #"/"))) v})))
                      _ (swap! db/tracker assoc-in
                          [flow-id ff]
                          {:start    start
                           :in-chan? (or (ut/chan? data-val) (try (some ut/chan? (vals data-val)) (catch Throwable _ false)))})
                      expression
                        (delay
                          (try
                            (cond (= data-val :skip)                           (async/timeout 100) ; :skip
                                  (or repeat-input? (ut/chan? data-val))       nil ;:skip ;[]
                                  (not pre-when?)                              [:failed-pre-check :w-input pre-in] ;; failed
                                  (and fully-resolved? is-subflow?) ;(and is-subflow?
                                    (let [a (atom nil)] ;; use temp atom to hold subflow results
                                      (flow block-function
                                            (merge (select-keys opts-map [:increment-id?])
                                                   {:flow-id        (str flow-id "/" (ut/unkeyword ff))
                                                    :parent-run-id  (get opts-map :run-id) ;; pass
                                                    :close-on-done? true ;; hardcoding for now,
                                                    :client-name    (get opts-map :client-name) ;; pass
                                                    :debug?         debug?})
                                            a ;output-chs
                                            subflow-overrides)
                                      (while (nil? @a) ;; possible out of control loop with a bad
                                        (Thread/sleep 100))
                                      @a)
                                  (and fully-resolved? is-accumulating-ports?) (apply (get block-function :fn) resolved-inputs)
                                  block-fn-raw?                                (block-function data-val)
                                  block-is-port?                               {:port-in? true ff data-val} ;; just
                                  :else                                        ((get block-function :fn) data-val))
                            (catch Throwable e ;Exception e ;; evaling user code here, want to
                              {:error (str e) :with-inputs (if resolved-inputs (vec resolved-inputs) data-val)})))
                      {:keys [result fn-start fn-end elapsed-ms]} (ut/timed-exec @expression timeout)
                      condis (into {}
                                   (for [[path f] (get block-function :cond)
                                         :let     [res (true? (try (f result) (catch Exception _ false)))
                                                   _ (swap! db/status assoc-in [flow-id :condis path] (true? res))]]
                                     {path res}))
                      not-condi-path (for [[k v] condis :when (false? v)] k)
                      condi-channels (vec (remove nil? (for [c (keys condis)] (get-in @db/condi-channels-atom [flow-id [ff c]])))) ;; all
                      not-condi-channels (vec (remove nil?
                                                (for [c not-condi-path] (get-in @db/condi-channels-atom [flow-id [ff c]])))) ;; only
                      error? (or (get result :error) (get result :timeout))
                      post-when? (try ((get block-function :post-when? (fn [_] true)) result) (catch Exception _ true)) ;; default
                                                                                                                        ;; to
                                                                                                                        ;; true
                                                                                                                        ;; on
                                                                                                                        ;; error?
                      view (if error? ;; if error, we'll force a view to better surface it in
                             (fn [x] [:box :child (str (get x :error))])
                             (get block-function :view))
                      view-out (when (fn? view)
                                 (try (let [vv (view result)] ;; if view isn't hiccup and just
                                        (if (string? vv) [:box :child (str vv)] vv))
                                      (catch Exception e
                                        (merge {:cant-eval-view-struct e :block ff}
                                               {:exception-data (ex-data e)}
                                               {:exception-details (Throwable->map e)}))))
                      _ (when (fn? view)
                          (swap! db/results-atom assoc-in
                            [flow-id (keyword (str (cstr/replace (str ff) ":" "") "-vw"))]
                            view-out))
                      web-val-map ;(merge
                        (cond (and fully-resolved? is-accumulating-ports?) {:v     (ut/limited result flow-id) ;; since
                                                                                                               ;; it's
                                                                                                               ;; going
                                                                                                               ;; to
                                                                                                               ;; the
                                                                            :input (ut/limited resolved-inputs flow-id)}
                              (and block-is-port? (ut/namespaced? ff)) ;; only true input ports
                                {:v       (ut/limited data-val flow-id)                                        ;; since
                                                                                                               ;; it's
                                                                                                               ;; going
                                                                                                               ;; to
                                 :port-of (first next-hops)}
                              :else                                        {:v     (ut/limited result flow-id) ;; since
                                                                                                               ;; it's
                                                                                                               ;; going
                                                                                                               ;; to
                                                                            :input (ut/limited data-val flow-id)})
                      value-only (fn [x] (if (and (map? x) (get x :port-in?)) (first (vals (dissoc x :port-in?))) x))
                      _ (when (and (not (ut/chan? result)) (not (ut/chan? data-val)))
                          (swap! db/resolved-paths assoc flow-id (conj (get @db/resolved-paths flow-id) the-path))
                          (swap! db/results-atom assoc-in [flow-id ff] (ut/limited result flow-id)) ;; final
                          (swap! db/fn-history assoc
                            flow-id
                            (conj (get @db/fn-history flow-id [])
                                  {:block        ff
                                   :from         data-from
                                   :value        (ut/limited (value-only result) flow-id)
                                   :type         :function
                                   :view         view-out
                                   :run-id       (get opts-map :run-id)
                                   :base-flow-id (str base-flow-id)
                                   :dest         ff
                                   :channel      [ff]
                                   :data-type    (ut/data-typer (ut/limited (get web-val-map :v) flow-id))
                                   :start        fn-start
                                   :end          fn-end
                                   :elapsed-ms   elapsed-ms}))
                          (swap! db/tracker assoc-in [flow-id ff :end] (System/currentTimeMillis)) ;; test
                          (when ;(and
                            (= ff :done)
                            (swap! db/status assoc-in [flow-id :end] (System/currentTimeMillis)) ;; test
                          )) ;))
                      output-chs (remove nil?
                                   (cond ;; output channel, condis, or regular output switch
                                     (or (not pre-when?) (not post-when?))       [] ;; failed
                                                                                    ;; post/pre-when,
                                                                                    ;; send nowhere.
                                     (and done-channel-push? (seqable? done-ch)) done-ch
                                     done-channel-push?                          [done-ch]
                                     #_{:clj-kondo/ignore [:not-empty?]}
                                     (not (empty? condi-channels))               (into condi-channels output-chs)
                                     :else                                       output-chs))]
                  (try (when #_{:clj-kondo/ignore [:not-empty?]}
                         (not (empty? output-chs))
                         (if (seq output-chs)
                           (doseq [oo   output-chs
                                   :let [result (if (some #(= oo %) not-condi-channels) (async/timeout 100) result)]] ;; forward
                                                                                                                      ;; processed
                                                                                                                      ;; data to
                                                                                                                      ;; the
                             (try (async/>! oo
                                            (if done-channel-push?
                                              result ;; no map wrapper for return channel
                                              {:sender ff :value result}))
                                  (catch Exception e
                                    (ut/ppln [:caught-in-push-loop e :outputs (count output-chs) :this oo :output-chs
                                              output-chs]))))
                           (async/>! output-chs
                                     {:sender ff
                                      :value  (if (some #(= output-chs %) not-condi-channels) (async/timeout 100) result)}))) ;)
                       (catch Exception e (ut/ppln [:caught-in-doseq-push-loop e :output-chs output-chs])))
                  (recur pending-chs)))
              (recur pending-chs)))))
      (catch Throwable e (ut/ppln [:main-go-loop-in-process-error! e])))))

(defn trigger-guard
  [flow-id]
  (let [max-iterations 100000] ;; so we dont loop forever in case something gets effed
    (loop [iteration 0]
      (when (and (< iteration max-iterations) (contains? @db/trigger-guard flow-id))
        (Thread/sleep 500)
        (ut/ppln [:trigger-guard :waiting-until-previous-run-shuts-down flow-id iteration])
        (recur (inc iteration))))))

(defn flow
  [network-map & [opts-map done-ch subflow-map]]
  (try ;; to pretty print the exception data only and not tons of other unhelpful garbage
    (cond ;; check spec, was instrumenting but the error msgs were horrible
      (not (s/valid? ::network-map network-map)) (throw (ex-info "Invalid network-map"
                                                                 {:exception      :flow-cannot-run!
                                                                  :failed-spec-on :flow-map
                                                                  :issue          (s/explain-str ::network-map network-map)
                                                                  :input          network-map}))
      (and opts-map (not (s/valid? ::opts-map opts-map))) (throw (ex-info "Invalid opts-map"
                                                                          {:exception      :flow-cannot-run!
                                                                           :failed-spec-on :flow-opts-map
                                                                           :issue          (s/explain-str ::opts-map opts-map)
                                                                           :input          opts-map}))
      (and done-ch (not (s/valid? ::done-ch done-ch))) (throw (ex-info "Invalid done-ch"
                                                                       {:exception      :flow-cannot-run!
                                                                        :failed-spec-on :output-channel-or-atom
                                                                        :issue          (s/explain-str ::done-ch done-ch)}))
      (and subflow-map (not (s/valid? ::subflow-map subflow-map))) (throw (ex-info "Invalid subflow-map"
                                                                                   {:exception      :flow-cannot-run!
                                                                                    :failed-spec-on :subflow-override-map
                                                                                    :issue          (s/explain-str ::subflow-map
                                                                                                                   subflow-map)
                                                                                    :input          subflow-map}))
      :else
        (let [flow-id       (or (get opts-map :flow-id) (ut/generate-name))
              _ (trigger-guard flow-id) ;; this is a blocker in case someone starts a flow
              increment-id? (get opts-map :increment-id? true)
              base-flow-id  flow-id ;; saving for later before below mutation
              run-id        (str (java.util.UUID/randomUUID))
              parent-run-id (get opts-map :parent-run-id)
              flow-id       (if increment-id? ;(not (= flow-id "live-scratch-flow"))
                              (str flow-id "-" (count (filter #(cstr/starts-with? % flow-id) (keys @db/channel-history))))
                              flow-id)] ;; don't number the scratch-flow iterations...
          (swap! db/results-atom dissoc flow-id)
          (swap! db/channel-history dissoc flow-id)
          (swap! db/fn-history dissoc flow-id)
          (swap! db/block-dump dissoc flow-id)
          (swap! db/tracker dissoc flow-id)
          (swap! db/status dissoc flow-id)
          (swap! db/waffle-data dissoc flow-id)
          (swap! db/port-accumulator dissoc flow-id)
          (swap! db/block-defs dissoc flow-id) ;; save and ship these in case the UI somehow
          (when #_{:clj-kondo/ignore [:not-empty?]} ;; clean up old channels on new run, if
            (and (not (empty? (get @db/channels-atom flow-id))) ;; - we have channels
                 (not (true? @web/websocket?))      ;; - web ui is NOT running
                 (not (get opts-map :debug? true))) ;; - debug is NOT enabled
            (doseq [[k c] (get @db/channels-atom flow-id)]
              (try #_{:clj-kondo/ignore [:redundant-do]}
                   (do (when (get opts-map :debug? false) (ut/ppln [:closing-channel k c])) (async/close! c))
                   (catch Exception e (ut/ppln [:error-closing-channel e k c])))))
          (try
            (let [{:keys [components connections]} network-map
                  opts-map                         (merge {:debug? true :debux? false :close-on-done? false :run-id run-id}
                                                          opts-map) ;; merge
                  {:keys [debug? debux?]}          opts-map
                  pp                               (if debug? ut/ppln ut/pplno) ;; TODO, gross.
                  web-push?                        (true? @web/websocket?)
                  gen-connections                  (for [[_ f] connections
                                                         :let  [base (ut/unkeyword (first (cstr/split (str f) #"/")))]
                                                         :when (cstr/includes? (str f) "/")]
                                                     [f (keyword base)])
                  components                       (into {} ;; todo, integrate this into the logic
                                                         (for [[k v] components] ;; <- sub-flow
                                                           {k (if (s/valid? ::network-map v)
                                                                (merge v
                                                                       {:inputs (vec (for [c (filter #(cstr/includes? (str %)
                                                                                                                      (str k "/"))
                                                                                               (distinct (flatten connections)))]
                                                                                       (keyword (last (cstr/split (str c)
                                                                                                                  #"/")))))})
                                                                v)}))
                  named-connections                connections
                  connections                      (vec (distinct (into connections gen-connections)))
                  channels                         (into {}
                                                         (for [conn connections] ;; TODO
                                                           {conn (async/chan 1)}))
                  components                       (assoc (merge ;; filter out comps that have no
                                                            (merge
                                                              (into {} (for [[_ v] components :when (get v :cond)] (get v :cond)))
                                                              (into {}
                                                                    (flatten (for [[k v] components
                                                                                   :let  [ins (get v :inputs)]
                                                                                   :when ins]
                                                                               (for [i ins]
                                                                                 {(keyword
                                                                                    (str (ut/unkeyword k) "/" (ut/unkeyword i)))
                                                                                    {:fn :port}})))))
                                                            components)
                                                     :done {:fn (fn [x] x)}) ;; add in 'implied'
                  condi-connections                (vec (apply concat
                                                          (for [[k v] components
                                                                :when (get v :cond)]
                                                            (for [c (keys (get v :cond))] [k c]))))
                  condi-channels                   (into {} (for [conn condi-connections] {conn (async/chan 1)}))
                  connections                      (vec (distinct (into connections condi-connections))) ;; add
                  coords-map                       (ut/coords-map connections)
                  fp                               (path-map connections)
                  sources                          (fn [to]
                                                     (vec (distinct (for [[ffrom tto] connections :when (= tto to)] ffrom))))
                  components                       (select-keys components (vec (distinct (flatten (apply conj connections))))) ;; no
                  input-mappings                   (into {} ;; mostly for web
                                                         (for [b (keys components)]
                                                           {b (vec (remove #{b}
                                                                     (flatten (filter #(= (second %) b) connections))))}))
                  looper                           (vec (distinct (apply concat
                                                                    (for [m (vec (for [e fp] (frequencies (flatten (second e)))))]
                                                                      (for [[k v] m :when (> v 1)] k)))))
                  started                          (atom #{})]
              (swap! db/status assoc-in [flow-id :start] (System/currentTimeMillis))
              (swap! db/subflow-overrides assoc-in [flow-id run-id] subflow-map)
              (when parent-run-id (swap! db/results-atom assoc-in [flow-id :parent-run-id] parent-run-id))
              (swap! db/results-atom assoc-in [flow-id :run-id] run-id)     ;; uid for this run
              (swap! db/results-atom assoc-in [flow-id :opts-map] opts-map) ;; extra arbitrary
              (when web-push? (rest/push! flow-id :none [flow-id :blocks] {}))
              (swap! db/working-data assoc
                flow-id ;; network-map
                {:connections     named-connections
                 :points          (get network-map :points)
                 :hide            (get network-map :hide)
                 :limit           (get network-map :limit db/sample-limit)
                 :rewinds         (get network-map :rewinds db/rewinds)
                 :description     (get network-map :description)
                 :colors          (get network-map :colors :Paired)
                 :docs            (into {}
                                        (remove empty?
                                          (for [[k {:keys [doc]}] (get network-map :components) :when (not (nil? doc))] {k doc})))
                 :gen-connections gen-connections
                 :components-list (keys components)
                 :components      (pr-str components)
                 :condis          condi-connections})
              (pp [:gen-connections gen-connections :connections connections :fp fp :looper looper :comps components :coords-map
                   coords-map :condi-connections condi-connections])
              (when web-push? (rest/push! flow-id :none [flow-id :network-map] (get @db/working-data flow-id)))
              (when web-push?
                #_{:clj-kondo/ignore [:redundant-do]} ;; lol, it is, in fact, NOT redundant.
                (do
                  (doseq [[bid v] components
                          :when   (not (= bid :done))
                          :let    [from-val (get components bid) ;; incoming fn or static -
                                   static-input? (and (not (fn? from-val)) ;; not a raw fn block
                                                      (nil? (get from-val :fn)))
                                   is-subflow? (s/valid? ::network-map from-val)
                                   view-mode (get-in network-map [:canvas bid :view-mode])
                                   hidden1? (get-in network-map [:canvas bid :hidden?])
                                   hidden? (if (and (nil? hidden1?) (ut/namespaced? bid)) true hidden1?)
                                   block-def
                                     (merge
                                       {:y                 (or (get-in network-map [:canvas bid :y])
                                                               (get v :y (get-in coords-map [bid :y]))) ;; deprecated
                                        :x                 (or (get-in network-map [:canvas bid :x])
                                                               (get v :x (get-in coords-map [bid :x]))) ;; deprecated
                                        :base-type         :artifacts
                                        :width             (or (get-in network-map [:canvas bid :w]) (get v :w 240)) ;; deprecated
                                        :height            (or (get-in network-map [:canvas bid :h]) (get v :h 255)) ;; deprecated
                                        :type              :text
                                        :flowmaps-created? true
                                        :block-type        "map2"
                                        :hidden?           hidden?
                                        :hidden-by         (when (ut/namespaced? bid)
                                                             (try (edn/read-string (first (cstr/split (str bid) #"/")))
                                                                  (catch Exception _ nil)))
                                        :view-mode         (if (get v :view) "view" "data")
                                        :options           [{:prefix        ""
                                                             :suffix        ""
                                                             :block-keypath [:view-mode]
                                                             :values        (vec (remove nil?
                                                                                   ["data" (when (get v :view) "view")
                                                                                    (when (get v :doc) "doc") "grid" "text"
                                                                                    (when (and (not static-input?) debux?) "dbgn")
                                                                                    (when (and (not is-subflow?) static-input?)
                                                                                      "input")]))}]}
                                       (when is-subflow? {:subflow? true})
                                       (when static-input?
                                         {:text-input? true ;; override the view mode w input
                                          :view-mode   "input"})
                                       (when view-mode {:view-mode view-mode}))]]
                    (do (swap! db/block-defs assoc-in [flow-id bid] block-def)
                        (rest/push! flow-id bid [flow-id :blocks bid] block-def)))
                  (doseq [[bid v] (get network-map :canvas) ;; decorative front-end-only blocks
                          :when   (string? bid)
                          :let    [block-def {:y          (or (get-in network-map [:canvas bid :y])
                                                              (get v :y (get-in coords-map [bid :y]))) ;; deprecated
                                              :x          (or (get-in network-map [:canvas bid :x])
                                                              (get v :x (get-in coords-map [bid :x]))) ;; deprecated
                                              :base-type  :artifacts
                                              :width      (or (get-in network-map [:canvas bid :w]) (get v :w 240)) ;; deprecated
                                              :height     (or (get-in network-map [:canvas bid :h]) (get v :h 255)) ;; deprecated
                                              :type       :text
                                              :inputs     (get-in network-map [:canvas bid :inputs])
                                              :block-type "text"}]]
                    (do (swap! db/block-defs assoc-in [flow-id bid] block-def)
                        (rest/push! flow-id bid [flow-id :blocks bid] block-def)))
                  (doseq [[bid inputs] input-mappings
                          :when        (not (= bid :done))
                          :let         [inputs (vec (conj (vec (for [i inputs] [i [:text [:no nil]]]))
                                                          [nil [:text [:paragraph 0]]]))]] ;; not
                    (do (swap! db/block-defs assoc-in [flow-id bid :inputs] inputs)
                        (rest/push! flow-id bid [flow-id :blocks bid :inputs] inputs)))))
              (swap! db/channels-atom assoc flow-id channels)
              (swap! db/condi-channels-atom assoc flow-id condi-channels)
              (doseq [c connections] ;;; "boot" channels into history even if they never get
                (swap! db/channel-history update
                  flow-id
                  conj
                  {:path         [:creating-channels :*]
                   :channel      c
                   :base-flow-id (str base-flow-id)
                   :run-id       (get opts-map :run-id)
                   :start        (System/currentTimeMillis)
                   :end          (System/currentTimeMillis)
                   :value        nil
                   :data-type    "boot"}))
              (doseq [from (keys components)]
                (let [start             (System/currentTimeMillis)
                      subflow-override? (not (nil? (get subflow-map from)))
                      fn-is-subflow?    (s/valid? ::network-map (get components from))
                      mod-channel-keys  (fn [x] (for [[f t] x] [f flow-id t]))
                      from-val          (if subflow-override? (get subflow-map from) (get components from))
                      out-conns         (filter #(= from (first %)) connections)
                      in-conns          (filter #(= from (last %)) connections)
                      in-condi-conns    (filter #(= from (last %)) condi-connections)
                      to-chans          (map channels (filter #(= from (first %)) connections))]
                  (pp (str "  ... Processing: " from))
                  (if (and (not (fn? from-val)) ;; not a raw fn
                           (nil? (get from-val :fn))
                           (not fn-is-subflow?)) ;; make sure its not a sub-flow map / we will
                    (let [text-input? (true? (some #(= :text-input %) (flatten [from-val])))
                          from-val    (if text-input? (last from-val) from-val)
                          _ (swap! db/tracker assoc-in [flow-id from] {:start start :end start :static? true})]
                      (when (some #(= % from) (get network-map :hide))
                        (swap! db/hidden-values assoc-in [flow-id from-val] "***HIDDEN-FROM-UI***"))
                      (when web-push?
                        (rest/push! flow-id from [flow-id :blocks from :body] (str {:v (ut/limited from-val flow-id)})))
                      (pp (vec (remove nil?
                                 [:block from :pushing-static-value from-val :to (vec (map last out-conns))
                                  (when subflow-override? :subflow-value-override!)])))
                      (swap! started conj from)
                      (swap! db/results-atom assoc-in [flow-id from] from-val)
                      (when web-push? ;; seeding the block history with this static value, just
                        (rest/push! flow-id
                                    from
                                    [flow-id :blocks from :body]
                                    {:v (ut/limited from-val flow-id)}
                                    start
                                    (System/currentTimeMillis)))
                      (swap! db/fn-history assoc
                        flow-id
                        (conj (get @db/fn-history flow-id [])
                              {:block        from
                               :from         :static
                               :path         [:from :static from]
                               :value        (ut/limited from-val flow-id)
                               :base-flow-id (str base-flow-id)
                               :type         :function
                               :run-id       (get opts-map :run-id)
                               :dest         from
                               :channel      [from]
                               :data-type    (ut/data-typer (ut/limited from-val flow-id))
                               :start        start
                               :end          (System/currentTimeMillis)
                               :elapsed-ms   (- (System/currentTimeMillis) start)})) ;; temp
                      (doseq [c to-chans] (async/put! c {:sender from :value from-val})))
                    (let [;in-chans (map channels in-conns)
                          in-chans     (remove nil? (into (map channels in-conns) (map condi-channels in-condi-conns)))
                          has-starter? (get from-val :starter)
                          out-chans    (map channels out-conns)
                          from-val     (if (map? from-val) ;(and (map? from-val) (not (s/valid?
                                         (merge from-val
                                                {:inputs ;(if (s/valid? ::network-map from-val)
                                                   (vec (for [i (get from-val :inputs)]
                                                          (keyword (str (ut/unkeyword from) "/" (ut/unkeyword i)))))})
                                         from-val)
                          srcs         (sources from)]
                      (pp ["Creating channels for" from :-> srcs :already-started (count (set @started)) :blocks])
                      (let [] ;do ;when parents-ready? ;; placeholder
                        (pp ["Starting block " from " with channel(s): " in-chans])
                        (swap! started conj from)
                        (async/thread (pp ["In thread for " from " reading from channel " in-chans])
                                      (process flow-id
                                               base-flow-id
                                               connections
                                               in-chans
                                               out-chans
                                               from
                                               fp
                                               from-val
                                               opts-map
                                               done-ch
                                               components)))
                      (when has-starter? ;; 95% dupe code from above - refactor TODO - used to
                        (let [;text-input? false ;(true? (some #(= :text-input %) (flatten
                              from-val (get from-val :starter)] ;; lazy override temp for this
                          (when web-push? (rest/push! flow-id from [flow-id :blocks from :body] (str {:v from-val})))
                          (pp (vec (remove nil?
                                     [:block-starter from :pushing-static-value from-val :to (vec (map last out-conns))
                                      (when subflow-override? :subflow-value-override!)])))
                          (swap! started conj from)
                          (swap! db/results-atom assoc-in [flow-id from] from-val)
                          (when web-push? ;; seeding the block history with this static value,
                            (rest/push! flow-id from [flow-id :blocks from :body] {:v from-val} start (System/currentTimeMillis)))
                          (swap! db/fn-history assoc
                            flow-id
                            (conj (get @db/fn-history flow-id [])
                                  {:block        from
                                   :from         :starter
                                   :path         [:from :starter from]
                                   :value        (ut/limited from-val flow-id)
                                   :base-flow-id (str base-flow-id)
                                   :type         :function
                                   :run-id       (get opts-map :run-id)
                                   :dest         from
                                   :channel      [from]
                                   :data-type    (ut/data-typer (ut/limited from-val flow-id))
                                   :start        start
                                   :end          (System/currentTimeMillis)
                                   :elapsed-ms   (- (System/currentTimeMillis) start)})) ;; temp
                          (doseq [c to-chans] (async/put! c {:sender from :value from-val})))))))))
            (catch Exception e (ut/ppln {:flow-error (Throwable->map e)})))))
    (catch clojure.lang.ExceptionInfo e (ut/ppln (.getData e)))))


(defmethod wl/handle-request :push-live-flow
  [{:keys [value canvas]}]
  (try ;; live eval and send it (needs access to above fns w/o circ-loop) TODO refactor out
    (let [evl           (eval ;; TODO default to *this* namespace
                          (read-string value))
          evl-canvas    (get evl :canvas {})
          merged-canvas (merge evl-canvas canvas)
          evl           (assoc evl :canvas merged-canvas)
          valid?        (s/valid? ::network-map evl)
          spec-demangle {:issue (s/explain-str ::network-map evl)}]
      (when valid?
        (try ;; no point to try and run a bogus flow
          (flow evl {:flow-id "live-scratch-flow"})
          (catch Exception e (ut/ppln {:live-flow-exec-error e}))))
      (ut/ppln [:incoming-live-flow canvas evl-canvas merged-canvas value evl])
      [:live-flow-return [:live-flow-return]
       (merge {:sent value :received (System/currentTimeMillis) :return (pr-str evl) :valid? valid?}
              (when (not valid?) spec-demangle))])
    (catch Exception e
      (do (ut/ppln [:incoming-live-flow :error value (str e)])
          [:live-flow-return [:live-flow-return] {:error (str e) :received (System/currentTimeMillis)}]))))

(defn flow-results
  [] ;; somewhat pointless, but hey. TODO
  (into {}
        (for [flow-id (keys @db/resolved-paths)]
          {flow-id (let [res (dissoc (into {}
                                           (for [p (vec (distinct (get @db/resolved-paths flow-id)))]
                                             {p (for [pp p] [pp (get-in @db/results-atom [flow-id pp])])}))
                               [:error-path-issue!])] ;; TODO
                     res)})))

(defn ask [flow-id point-name value])



(defn schedule!
  [time-seq1 flowmap & args]
  (let [[opts chan-out override] args
        opts                     (if (nil? opts) {} opts)
        times                    (if (and (vector? time-seq1) (keyword? (first time-seq1)))
                                   (doall (take 1000 (ut/time-seq time-seq1)))
                                   time-seq1)
        times-unlimited          (if (and (vector? time-seq1) (keyword? (first time-seq1))) (ut/time-seq time-seq1) time-seq1)
        ch                       (chime/chime-at times-unlimited ;; [] chime time seq,
                                                 (fn [time]
                                                   (let [opts (merge opts {:schedule-started (str time)})]
                                                     (flow flowmap opts chan-out override)))
                                                 {:on-finished   (fn [] (ut/ppln [:schedule-finished! opts time-seq1]))
                                                  :error-handler (fn [e] (ut/ppln [:scheduler-error e]))})] ;; if
    (swap! db/live-schedules conj
      {:flow-id    (get opts :flow-id "unnamed-flow-sched")
       :override   override
       :next-times (try (vec (map str times)) (catch Exception _ times))
       :schedule   (if (vector? time-seq1) time-seq1 [:custom-time-fn])
       :channel    ch})))

(defn unschedule!
  [flow-id]
  (let [schedule-to-remove (some #(when (= (:flow-id %) flow-id) %) @db/live-schedules)]
    (when schedule-to-remove
      (async/close! (:channel schedule-to-remove))
      (swap! db/live-schedules #(remove (fn [x] (= (:flow-id x) flow-id)) %)))))








