(ns elasticsearch-bolt.storm
  (:use [elasticsearch-bolt.fields :only [elasticsearch-output-fields]]
        [elasticsearch-bolt.core :only [index-bulk]]
        [backtype.storm clojure config log])
  (:require [clojurewerkz.elastisch.rest :as esr]))


(defn success? [m]
  (boolean (or (get-in m [:index :ok])
               (get-in m [:create :ok]))))

(defbolt elasticsearch-write
  elasticsearch-output-fields {:prepare true :params [uri]}
  [conf context collector]
  (let [_ (log-message "Connecting to " uri)
        conn (esr/connect! uri)]
    (bolt
      (execute
        [tuple]
        (let [{:keys [meta index doctype docs]} tuple
              {:keys [items] :as result} (index-bulk index doctype docs)
              successful? (every? true? (map success? items))]
          (if successful?
            (let [output [meta :success]]
              (log-message (format "Succeeded in indexing %s documents."
                                   (count items)))
              (emit-bolt! collector output :anchor tuple)
              (ack! collector tuple))
            (do
              (log-warn (format "Not every doc was indexed successfully for %s/%s! Check elasticsearch logs for more info. Failing tuple. Result: %s\n"
                                index doctype (with-out-str (clojure.pprint/pprint result))))
              (fail! collector tuple))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 'Constrcutors'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- get-or-throw [m k]
  (let [result (get m k :not-found)]
    (if (= result :not-found)
      (throw (Exception. (format "Key %s was not found in properties: %s" k m)))
      result)))

(defn make-elasticsearch-bolt [properties]
  (let [uri (get-or-throw properties "ELASTICSEARCH_URL")]
    (log-message "getting es uri: " uri)
    (elasticsearch-write uri)))
