(ns com.sixsq.nuvla.server.resources.user-template-github-lifecycle-test
  (:require
    [clojure.test :refer [deftest is use-fixtures]]
    [com.sixsq.nuvla.auth.external :as ex]
    [com.sixsq.nuvla.auth.github :as auth-github]
    [com.sixsq.nuvla.server.app.params :as p]
    [com.sixsq.nuvla.server.middleware.authn-info :as authn-info]
    [com.sixsq.nuvla.server.resources.callback.utils :as cbu]
    [com.sixsq.nuvla.server.resources.configuration :as configuration]
    [com.sixsq.nuvla.server.resources.configuration-nuvla :as config-nuvla]
    [com.sixsq.nuvla.server.resources.lifecycle-test-utils :as ltu]
    [com.sixsq.nuvla.server.resources.user :as user]
    [com.sixsq.nuvla.server.resources.user-template :as ut]
    [com.sixsq.nuvla.server.resources.user-template-github :as github]
    [com.sixsq.nuvla.server.resources.user-template-minimum :as minimum]
    [com.sixsq.nuvla.server.resources.user.user-identifier-utils :as uiu]
    [com.sixsq.nuvla.server.util.metadata-test-utils :as mdtu]
    [jsonista.core :as j]
    [peridot.core :refer [content-type header request session]]))


(use-fixtures :each ltu/with-test-server-fixture)


(def base-uri (str p/service-context user/resource-type))


(def configuration-base-uri (str p/service-context configuration/resource-type))


(def user-template-base-uri (str p/service-context ut/resource-type))


(def ^:const callback-pattern #".*/api/callback/.*/execute")


;; callback state reset between tests
(defn reset-callback! [callback-id]
  (cbu/update-callback-state! "WAITING" callback-id))


(def configuration-user-github {:template {:service       "session-github" ;;reusing configuration from session GitHub
                                           :instance      github/registration-method
                                           :client-id     "FAKE_CLIENT_ID"
                                           :client-secret "ABCDEF..."}})


(deftest check-metadata
  (mdtu/check-metadata-exists (str ut/resource-type "-" github/resource-url)
                              (str ut/resource-type "-" github/resource-url "-create")))


(deftest lifecycle

  (let [href                 (str ut/resource-type "/" github/registration-method)
        template-url         (str p/service-context ut/resource-type "/" github/registration-method)

        session-anon         (-> (ltu/ring-app)
                                 session
                                 (content-type "application/json")
                                 (header authn-info/authn-info-header "user/unknown user/unknown group/nuvla-anon"))
        session-admin        (header session-anon authn-info/authn-info-header "group/nuvla-admin group/nuvla-admin group/nuvla-user group/nuvla-anon")
        session-user         (header session-anon authn-info/authn-info-header "user/jane user/jane group/nuvla-user group/nuvla-anon")

        redirect-url-example "https://example.com/webui"]

    ;; must create the github user template; this is not created automatically
    (let [user-template (->> {:group  "my-group"
                              :icon   "some-icon"
                              :order  10
                              :hidden false}
                             (merge github/resource)
                             j/write-value-as-string)]

      (-> session-admin
          (request user-template-base-uri
                   :request-method :post
                   :body user-template)
          (ltu/is-status 201))

      (-> session-anon
          (request template-url)
          (ltu/body->edn)
          (ltu/is-status 200)
          (ltu/body)))

    ;; get user template so that user resources can be tested
    (let [name-attr            "name"
          description-attr     "description"
          tags-attr            ["a-1" "b-2"]

          href-create          {:name        name-attr
                                :description description-attr
                                :tags        tags-attr
                                :template    {:href href}}

          href-create-redirect {:template {:href         href
                                           :redirect-url redirect-url-example}}
          invalid-create       (assoc-in href-create [:template :invalid] "BAD")]

      ;; queries by anyone should succeed but have no entries
      (doseq [session [session-anon session-user session-admin]]
        (-> session
            (request base-uri)
            (ltu/body->edn)
            (ltu/is-status 200)
            (ltu/is-count zero?)))

      ;; configuration must have GitHub client ID or secret, if not should get 500
      (-> session-anon
          (request base-uri
                   :request-method :post
                   :body (j/write-value-as-string href-create))
          (ltu/body->edn)
          (ltu/message-matches #".*missing or incorrect configuration.*")
          (ltu/is-status 500))

      ;; create the session-github configuration to use for these tests
      (let [cfg-href (-> session-admin
                         (request configuration-base-uri
                                  :request-method :post
                                  :body (j/write-value-as-string configuration-user-github))
                         (ltu/body->edn)
                         (ltu/is-status 201)
                         (ltu/location))]

        (is (= cfg-href (str "configuration/session-github-" github/registration-method)))

        (let [resp (-> session-anon
                       (request base-uri
                                :request-method :post
                                :body (j/write-value-as-string href-create))
                       (ltu/body->edn)
                       (ltu/is-status 303))

              uri  (-> resp ltu/location)

              resp (-> session-anon
                       (request base-uri
                                :request-method :post
                                :body (j/write-value-as-string href-create-redirect))
                       (ltu/body->edn)
                       (ltu/is-status 303))
              uri2 (-> resp ltu/location)

              ;; when configured nuvla will not authorize not configured redirect urls
              _ (binding [config-nuvla/*authorized-redirect-urls* ["https://nuvla.io"]]
                  (-> session-anon
                      (request base-uri
                               :request-method :post
                               :body (j/write-value-as-string href-create-redirect))
                      (ltu/body->edn)
                      (ltu/is-status 400)))


              uris [uri uri2]]

          ;; redirect URLs in location header should contain the client ID and resource id
          (doseq [u uris]
            (is (re-matches #".*FAKE_CLIENT_ID.*" (or u "")))
            (is (re-matches callback-pattern (or u ""))))

          ;; anonymous, user and admin query should succeed but have no users
          (doseq [session [session-anon session-user session-admin]]
            (-> session
                (request base-uri)
                (ltu/body->edn)
                (ltu/is-status 200)
                (ltu/is-count zero?)))

          ;;
          ;; test validation callback
          ;;

          (let [get-redirect-url #(->> % (re-matches #".*redirect_uri=(.*)$") second)
                get-callback-id  #(->> % (re-matches #".*(callback.*)/execute$") second)

                validate-urls    (map get-redirect-url uris)
                callbacks        (map get-callback-id validate-urls)]


            ;; all callbacks must exist
            (doseq [callback callbacks]
              (-> session-admin
                  (request (str p/service-context callback))
                  (ltu/body->edn)
                  (ltu/is-status 200)))


            ;; remove the authentication configuration
            (-> session-admin
                (request (str p/service-context cfg-href)
                         :request-method :delete)
                (ltu/body->edn)
                (ltu/is-status 200))

            ;; try hitting the callback with an invalid server configuration
            ;; when a redirect-url is present, the return is 303 even on errors
            (doseq [[url status] (map vector validate-urls [500 303 303])]
              (-> session-anon
                  (request url
                           :request-method :get)
                  (ltu/body->edn)
                  (ltu/message-matches #".*missing or incorrect configuration.*")
                  (ltu/is-status status)))

            ;; add the configuration back again
            (-> session-admin
                (request configuration-base-uri
                         :request-method :post
                         :body (j/write-value-as-string configuration-user-github))
                (ltu/body->edn)
                (ltu/is-status 201))

            ;; try hitting the callback without the OAuth code parameter
            ;; when a redirect-url is present, the return is 303 even on errors
            (doseq [[callback url status] (map vector callbacks validate-urls [400 303 303])]
              (reset-callback! callback)
              (-> session-anon
                  (request url)
                  (ltu/body->edn)
                  (ltu/message-matches #".*not contain required code.*")
                  (ltu/is-status status)))

            ;; try hitting the callback with mocked codes
            (doseq [[callback url status create-status n] (map vector callbacks validate-urls [400 303 303] [201 303 303] (range))]
              (reset-callback! callback)

              (let [github-login (str "GITHUB_USER_" n)
                    email        (format "user-%s@example.com" n)]

                (with-redefs [auth-github/get-github-access-token (fn [_ _ oauth-code]
                                                                    (case oauth-code
                                                                      "GOOD" "GOOD_ACCESS_CODE"
                                                                      "BAD" "BAD_ACCESS_CODE"
                                                                      nil))
                              auth-github/get-github-user-info    (fn [access-code]
                                                                    (when (= access-code "GOOD_ACCESS_CODE")
                                                                      {:login github-login, :email email}))]

                  (-> session-anon
                      (request (str url "?code=NONE")
                               :request-method :get)
                      (ltu/body->edn)
                      (ltu/message-matches #".*unable to retrieve GitHub access code.*")
                      (ltu/is-status status))

                  (is (nil? (uiu/user-identifier->user-id :github "github" github-login)))

                  (reset-callback! callback)
                  (-> session-anon
                      (request (str url "?code=BAD")
                               :request-method :get)
                      (ltu/body->edn)
                      (ltu/message-matches #".*unable to retrieve GitHub user information.*")
                      (ltu/is-status status))

                  (is (nil? (uiu/user-identifier->user-id :github nil github-login)))

                  (reset-callback! callback)
                  (-> session-anon
                      (request (str url "?code=GOOD"))
                      (ltu/body->edn)
                      (ltu/is-status create-status))

                  (let [user-id     (uiu/user-identifier->user-id :github "github" github-login)
                        name-value  (uiu/generate-identifier :github "github" github-login)
                        user-record (ex/get-user user-id)]

                    (is (not (nil? user-id)))

                    (is (not (nil? user-record)))

                    (is (= name-value (:name user-record)))

                    ;; FIXME: Fix code to put in alternate method from 'minimum'.
                    (is (= minimum/registration-method (:method user-record))))

                  ;; try creating the same user again, should fail
                  (reset-callback! callback)
                  (-> session-anon
                      (request (str url "?code=GOOD")
                               :request-method :get)
                      (ltu/body->edn)
                      (ltu/message-matches #".*an account with the same email already exists.*")
                      (ltu/is-status status))))))


          ;; create with invalid template fails
          (-> session-anon
              (request base-uri
                       :request-method :post
                       :body (j/write-value-as-string invalid-create))
              (ltu/body->edn)
              (ltu/is-status 400)))))))
