(ns com.sixsq.nuvla.server.resources.vulnerability-lifecycle-test
  (:require
    [clojure.data.json :as json]
    [clojure.test :refer [deftest is use-fixtures]]
    [peridot.core :refer [content-type header request session]]
    [com.sixsq.nuvla.server.app.params :as p]
    [com.sixsq.nuvla.server.middleware.authn-info :refer [authn-info-header]]
    [com.sixsq.nuvla.server.resources.common.utils :as u]
    [com.sixsq.nuvla.server.resources.lifecycle-test-utils :as ltu]
    [com.sixsq.nuvla.server.resources.vulnerability :as t]
    [com.sixsq.nuvla.server.util.metadata-test-utils :as mdtu]))


(use-fixtures :once ltu/with-test-server-fixture)


(def base-uri (str p/service-context t/resource-type))


(def valid-acl-admin {:owners    ["group/nuvla-admin"]
                      :view-data ["group/nuvla-user"]})

(def valid-acl-user {:owners    ["group/nuvla-user"]
                     :view-data ["group/nuvla-user"]})


(deftest check-metadata
  (mdtu/check-metadata-exists t/resource-type))


(deftest lifecycle
  (let [session-anon              (-> (ltu/ring-app)
                                      session
                                      (content-type "application/json"))
        session-admin             (header session-anon authn-info-header "user/abcdef01-abcd-abcd-abcd-abcdef012347 user/abcdef01-abcd-abcd-abcd-abcdef012347 group/nuvla-admin group/nuvla-user group/nuvla-anon")
        session-user              (header session-anon authn-info-header "user/abcdef01-abcd-abcd-abcd-abcdef012346 user/abcdef01-abcd-abcd-abcd-abcdef012346 group/nuvla-user group/nuvla-anon")

        valid-vulnerability-admin {:name        "my-vulnerability"
                                   :description "my-vulnerability description"
                                   :acl         valid-acl-admin
                                   }

        valid-vulnerability-user  (assoc valid-vulnerability-admin :acl valid-acl-user :name "unacceptable user vuln")]

    ;; admin query succeeds but is empty
    (-> session-admin
        (request base-uri)
        (ltu/body->edn)
        (ltu/is-status 200)
        (ltu/is-count zero?)
        (ltu/is-operation-present :add)
        (ltu/is-operation-absent :delete)
        (ltu/is-operation-absent :edit))

    ;; anon query fails
    (-> session-anon
        (request base-uri)
        (ltu/body->edn)
        (ltu/is-status 403))

    ;; user query succeeds, but doesn't have add operation as admin
    (-> session-user
        (request base-uri)
        (ltu/body->edn)
        (ltu/is-status 200)
        (ltu/is-count zero?)
        (ltu/is-operation-absent :add)
        (ltu/is-operation-absent :delete)
        (ltu/is-operation-absent :edit))

    ;; anon create must fail
    (-> session-anon
        (request base-uri
                 :request-method :post
                 :body (json/write-str valid-vulnerability-admin))
        (ltu/body->edn)
        (ltu/is-status 403))

    ;; user create must fail
    (-> session-user
        (request base-uri
                 :request-method :post
                 :body (json/write-str valid-vulnerability-user))
        (ltu/body->edn)
        (ltu/is-status 403))

    ;; check vulnerability creation
    (let [admin-uri     (-> session-admin
                            (request base-uri
                                     :request-method :post
                                     :body (json/write-str valid-vulnerability-admin))
                            (ltu/body->edn)
                            (ltu/is-status 201)
                            (ltu/location))

          admin-abs-uri (str p/service-context admin-uri)]

      ;; admin should see 1 vulnerability resource
      (-> session-admin
          (request base-uri)
          (ltu/body->edn)
          (ltu/is-status 200)
          (ltu/is-resource-uri t/collection-type)
          (ltu/is-count 1))

      ;; user also sees 1
      (-> session-user
          (request base-uri)
          (ltu/body->edn)
          (ltu/is-status 200)
          (ltu/is-resource-uri t/collection-type)
          (ltu/is-count 1))

      ;; verify contents of admin vulnerability
      (let [vulnerability-full (-> session-admin
                                   (request admin-abs-uri)
                                   (ltu/body->edn)
                                   (ltu/is-status 200)
                                   (ltu/is-operation-present :edit)
                                   (ltu/is-operation-present :delete))
            vulnerability      (:body (:response vulnerability-full))]

        (is (= "my-vulnerability" (:name vulnerability)))

        ;; verify that an edit works
        (let [updated (assoc vulnerability :name "edit name")]

          (-> session-admin
              (request admin-abs-uri
                       :request-method :put
                       :body (json/write-str updated))
              (ltu/body->edn)
              (ltu/is-status 200)
              (ltu/body))

          (let [updated-body (-> session-admin
                                 (request admin-abs-uri)
                                 (ltu/body->edn)
                                 (ltu/is-status 200)
                                 (ltu/body))]

            (is (= "edit name" (:name updated-body))))))

      ;; user cannot delete the vulnerability
      (-> session-user
          (request admin-abs-uri :request-method :delete)
          (ltu/body->edn)
          (ltu/is-status 403))

      ;; admin can delete the vulnerability
      (-> session-admin
          (request admin-abs-uri :request-method :delete)
          (ltu/body->edn)
          (ltu/is-status 200)))))


(deftest bad-methods
  (let [resource-uri (str p/service-context (u/new-resource-id t/resource-type))]
    (ltu/verify-405-status [[resource-uri :post]])))
