(ns com.sixsq.nuvla.server.resources.hook-oidc-user
  "
Stripe oidc user.
"
  (:require
    [clojure.string :as str]
    [clojure.tools.logging :as log]
    [com.sixsq.nuvla.auth.external :as ex]
    [com.sixsq.nuvla.auth.oidc :as auth-oidc]
    [com.sixsq.nuvla.auth.utils.http :as uh]
    [com.sixsq.nuvla.auth.utils.sign :as sign]
    [com.sixsq.nuvla.server.app.params :as app-params]
    [com.sixsq.nuvla.server.resources.session-oidc.utils :as oidc-utils]
    [com.sixsq.nuvla.server.util.response :as r]))


(def ^:const action "oidc-user")

(defn register-user
  [{:keys [base-uri params] :as request} redirect-ui-url]
  (let [instance          (get params :instance oidc-utils/geant-instance)
        {:keys [client-id client-secret token-url jwks-url]} (oidc-utils/config-oidc-params redirect-ui-url instance)
        redirect-hook-url (cond-> (str base-uri "hook" "/" action)
                                  (not= instance oidc-utils/geant-instance) (str "/" instance))]
    (log/info "hook-oidc-user redirect request:" request)
    (if-let [code (uh/param-value request :code)]
      (if-let [id-token (auth-oidc/get-id-token
                          client-id client-secret token-url code redirect-hook-url)]
        (try
          (let [public-key (auth-oidc/get-public-key id-token jwks-url)
                {:keys [sub email] :as claims} (sign/unsign-cookie-info id-token public-key)]
            (log/debugf "oidc access token claims for %s: %s" instance (pr-str claims))
            (if sub
              (or
                (ex/create-user! :oidc {:instance       instance
                                        :external-id    sub
                                        :external-email (or email (str sub "@fake-email.com"))})
                (oidc-utils/throw-user-exists redirect-ui-url))
              (oidc-utils/throw-no-subject redirect-ui-url)))
          (catch Exception e
            (oidc-utils/throw-invalid-access-code (str e) redirect-ui-url)))
        (oidc-utils/throw-no-access-token redirect-ui-url))
      (oidc-utils/throw-missing-code redirect-ui-url))))


(defn execute
  [{:keys [base-uri] :as request}]
  (log/debug "Executing hook oidc user registration" request)
  (let [redirect-ui-url         (-> base-uri
                                    (str/replace
                                      (re-pattern (str app-params/service-context "$"))
                                      app-params/ui-context)
                                    (str "sign-up"))
        redirect-ui-url-success (str redirect-ui-url "?message=signup-validation-success")]
    (try
      (if-let [user-id (register-user request redirect-ui-url)]
        (r/map-response (format "'%s' created" user-id) 303 nil redirect-ui-url-success)
        (r/map-response "could not create OIDC user" 400))
      (catch Exception e
        (or (ex-data e) (throw e))))))
