package com.socialquantum.battleship.gameapi.internal.serializationutils;

import com.google.gson.*;
import com.socialquantum.battleship.gameapi.internal.dataobjects.gamestatus.GameStateDataStatus;

import java.lang.reflect.Type;
import java.util.Objects;

/**
 * Фабрика загрузки и сохранения объектов.
 * Created 29/06/17 13:33
 *
 * @author Vladimir Bogodukhov
 **/
public class SerializationDeserializationFactory {

    private static class GameStateDataStatusDeserializer
            implements JsonSerializer<GameStateDataStatus>, JsonDeserializer<GameStateDataStatus> {

        @Override
        public JsonElement serialize(GameStateDataStatus src, Type typeOfSrc, JsonSerializationContext context) {
            if (src == null) return null;
            return new JsonPrimitive(src.ident());
        }

        @Override
        public GameStateDataStatus deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
                throws JsonParseException {
            return GameStateDataStatus.elementByName(json.getAsString());
        }

    }

    private static final GsonBuilder GSON_BUILDER = configBuilder();


    /**
     * Загрузить объект типа из строки
     *
     * @param data  данные в строке
     * @param clazz тип к которому приводятся данные
     * @param <T> тип к которому приводятся данные
     * @return объект типа
     */
    public <T> T loadObject(String data, Class<T> clazz) {
        return GSON_BUILDER.create().fromJson(data, clazz);
    }


    /**
     * Сохранить данные объекта в строку.
     *
     * @param dataObject - объект данных
     * @return строка с данными
     */
    public String saveObject(Object dataObject) {
        Objects.requireNonNull(dataObject);
        return GSON_BUILDER.create().toJson(dataObject);
    }

    /*  ------------------------------------------------------------------------------------------------------------  */

    private static GsonBuilder configBuilder() {
        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(GameStateDataStatus.class, new GameStateDataStatusDeserializer());
        return gsonBuilder;
    }

}
