package com.socialquantum.battleship.gameapi.internal.utils;

import com.socialquantum.battleship.gameapi.internal.types.DefaultOperationResult;
import com.socialquantum.battleship.gameapi.internal.types.OperationResult;

import java.util.Objects;
import java.util.function.Function;

/**
 * Created 29/06/17 08:38
 *
 * @author Vladimir Bogodukhov
 **/
public class GameUtils {

    /**
     * Вспомогательный интерфейс получения данных.
     *
     * @param <T> -
     */
    @FunctionalInterface
    public interface FunctionThatThrows<T> {
        T execute() throws Throwable;
    }

    /**
     * Вспомогательный интерфейс получения данных.
     *
     * @param <T> -
     */
    @FunctionalInterface
    public interface FunctionThatThrowsWoResult<T> {
        void execute() throws Throwable;
    }

    /**
     * Метод получения объектов результата
     *
     * @param code        код http
     * @param result      код резульатат
     * @param successData данные успешного выполнения
     * @param errorData   данные неуспешного выполнения
     * @param <S> -
     * @param <E> -
     * @return резульат выполнения операции (всегда не null)
     */
    public static <S, E> OperationResult<S, E> result(int code, OperationResult.Result result, S successData, E errorData) {
        return new DefaultOperationResult<S, E>(code, result, successData, errorData);
    }


    /**
     * Обернуть выполнение метода и выбросить другое исключение.
     *
     * @param excClass    -
     * @param excProvider -
     * @param function    -
     * @param <R> -
     * @param <X> -
     * @return -
     * @throws X -
     */
    public static <R, X extends Throwable> R wrapException(
            Class<X> excClass,
            Function<Throwable, X> excProvider,
            FunctionThatThrows<R> function) throws X {
        Objects.requireNonNull(function);
        Objects.requireNonNull(excProvider);
        try {
            return function.execute();
        } catch (Throwable t) {
            throw excProvider.apply(t);
        }
    }

    /**
     * Обернуть выполнение метода и выбросить другое исключение.
     *
     * @param excClass    -
     * @param excProvider -
     * @param function    -
     * @param <X> тип исключения
     * @param <T> тип данного
     * @throws X исключение типа
     */
    public static <T, X extends Throwable> void wrapExceptionWoResult(
            Class<X> excClass,
            Function<Throwable, X> excProvider,
            FunctionThatThrowsWoResult<T> function) throws X {
        Objects.requireNonNull(function);
        Objects.requireNonNull(excProvider);
        try {
            function.execute();
        } catch (Throwable t) {
            throw excProvider.apply(t);
        }
    }

}
