package com.steammachine.propertyeditor;

import static com.steammachine.propertyeditor.utils.CommonUtils.check;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.StreamSupport.stream;

import com.steammachine.propertyeditor.baseeditors.BooleanProperty;
import com.steammachine.propertyeditor.baseeditors.ByteProperty;
import com.steammachine.propertyeditor.baseeditors.ClassProperty;
import com.steammachine.propertyeditor.baseeditors.DefaultPropertyEditors;
import com.steammachine.propertyeditor.baseeditors.EnumProperty;
import com.steammachine.propertyeditor.baseeditors.IntegerProperty;
import com.steammachine.propertyeditor.baseeditors.LongProperty;
import com.steammachine.propertyeditor.baseeditors.PrimitiveBooleanProperty;
import com.steammachine.propertyeditor.baseeditors.PrimitiveByteProperty;
import com.steammachine.propertyeditor.baseeditors.PrimitiveIntegerProperty;
import com.steammachine.propertyeditor.baseeditors.PrimitiveLongProperty;
import com.steammachine.propertyeditor.baseeditors.PrimitiveShortProperty;
import com.steammachine.propertyeditor.baseeditors.ShortProperty;
import com.steammachine.propertyeditor.baseeditors.StringProperty;
import com.steammachine.propertyeditor.types.PropertyEditor;
import com.steammachine.propertyeditor.types.PropertyEditorFactory;
import com.steammachine.propertyeditor.types.PropertyEditors;
import com.steammachine.propertyeditor.types.Selection;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import lombok.AllArgsConstructor;
import lombok.Data;

public class EditorsUtils {

    @Data
    @AllArgsConstructor
    private static class SelectionItem {

        final String propertyName;
        final Class<?> paramType;
    }

    @Data
    @AllArgsConstructor
    static class PropertyDefinition {

        private final String name;
        private final boolean readOnly;
        private final Class propertyType;
    }

    @Data
    @AllArgsConstructor
    private static class PropertyInfo {

        PropertyDefinition propertyDefinition;
        PropertyEditorFactory propertyEditorFactory;
    }

    private static final Map<Class, PropertyEditorFactory> EDITORS = new HashMap<>();

    static {
        register(Class.class, ClassProperty::new);
        register(String.class, StringProperty::new);
        register(Enum.class, EnumProperty::new);

        register(Boolean.TYPE, PrimitiveBooleanProperty::new);
        register(Boolean.class, BooleanProperty::new);
        register(Byte.TYPE, PrimitiveByteProperty::new);
        register(Byte.class, ByteProperty::new);
        register(Short.TYPE, PrimitiveShortProperty::new);
        register(Short.class, ShortProperty::new);
        register(Integer.TYPE, PrimitiveIntegerProperty::new);
        register(Integer.class, IntegerProperty::new);
        register(Long.TYPE, PrimitiveLongProperty::new);
        register(Long.class, LongProperty::new);

    }

    private static <T> void register(Class<T> type, PropertyEditorFactory<T> factory) {
        EDITORS.put(type, factory);
    }

    private EditorsUtils() {

    }

    public static PropertyEditors getEditors(Selection selection) {
        return new DefaultPropertyEditors(collectProperties(selection).stream()
                .filter(propertyDefinition ->
                        getPropertyEditorFactory(propertyDefinition.getPropertyType()) != null)
                .map(propertyDefinition -> new PropertyInfo(propertyDefinition,
                        getPropertyEditorFactory(propertyDefinition.getPropertyType())))
                .map(t -> t.getPropertyEditorFactory().newPropertyEditor(
                        t.getPropertyDefinition().getName(),
                        t.getPropertyDefinition().getPropertyType(),
                        selection))
                .sorted(Comparator.comparing(PropertyEditor::propertyName))
                .collect(toList()));
    }

    static Set<PropertyDefinition> collectProperties(Selection selection) {
        check(selection.hasElements(), IllegalStateException::new);

        Map<String, SelectionItem> getters = reduceProps(stream(selection.spliterator(), false)
                .map(object -> stream(object.getClass().getMethods())
                        .filter(method -> method.getReturnType() != Void.TYPE)
                        .filter(method -> method.getParameterCount() == 0)
                        .filter(method -> !Modifier.isStatic(method.getModifiers()))
                        .filter(method -> Modifier.isPublic(method.getModifiers()))
                        .filter(method -> !Modifier.isAbstract(method.getModifiers()))
                        .filter(EditorsUtils::isGetter)
                        .filter(method -> getterPropName(method).length() > 0)
                        .map(method -> new SelectionItem(getterPropName(method),
                                method.getReturnType()))
                        .map(selectionItem -> {
                            Map<String, SelectionItem> map = new HashMap<>();
                            map.put(selectionItem.getPropertyName(), selectionItem);
                            return map;
                        })
                        .reduce(new HashMap<>(), EditorsUtils::mergeMaps))
                .collect(toList()));

        Map<String, SelectionItem> setters = reduceProps(stream(selection.spliterator(), false)
                .map(object -> stream(object.getClass().getMethods())
                        .filter(method -> method.getReturnType() == Void.TYPE)
                        .filter(method -> method.getParameterCount() == 1)
                        .filter(method -> !Modifier.isStatic(method.getModifiers()))
                        .filter(method -> Modifier.isPublic(method.getModifiers()))
                        .filter(method -> !Modifier.isAbstract(method.getModifiers()))
                        .filter(EditorsUtils::isSetter)
                        .filter(method -> setterPropName(method).length() > 0)
                        .map(method -> new SelectionItem(setterPropName(method),
                                method.getParameterTypes()[0]))
                        .map(selectionItem -> {
                            Map<String, SelectionItem> map = new HashMap<>();
                            map.put(selectionItem.getPropertyName(), selectionItem);
                            return map;
                        })
                        .reduce(new HashMap<>(), EditorsUtils::mergeMaps))
                .collect(toList()));

        return getters.values()
                .stream()
                .map(item -> new PropertyDefinition(item.getPropertyName(),
                        !setters.containsKey(item.getPropertyName()), item.getParamType()))
                .collect(toSet());
    }

    private static Map<String, SelectionItem>
    reduceProps(List<Map<String, SelectionItem>> properties) {
        check(!properties.isEmpty(), IllegalStateException::new);
        Map<String, SelectionItem> result = properties.get(0);
        for (int i = 1; i < properties.size(); i++) {
            Map<String, SelectionItem> current = properties.get(i);
            result.keySet().retainAll(current.keySet());
            for (Entry<String, SelectionItem> entry : current.entrySet()) {
                SelectionItem selectionItem = result.get(entry.getKey());
                if (selectionItem != null &&
                        !typesCompatible(selectionItem.paramType, entry.getValue().paramType)) {
                    current.remove(entry.getKey());
                }
            }
        }
        return result;
    }

    private static boolean typesCompatible(Class<?> paramType, Class<?> paramType1) {
        return paramType == paramType1;
    }

    private static Map<String, SelectionItem> mergeMaps(Map<String, SelectionItem> map,
            Map<String, SelectionItem> map2) {
        Map<String, SelectionItem> result = new HashMap<>(map);
        result.putAll(map2);
        return result;
    }

    public static boolean isGetter(Method method) {
        return (method.getReturnType() == Boolean.TYPE && method.getName()
                .startsWith("is")) || method.getName().startsWith("get");
    }

    public static boolean isSetter(Method method) {
        return method.getName().startsWith("set");
    }


    private static PropertyEditorFactory getPropertyEditorFactory(Class paramType) {
        while (paramType != null) {
            PropertyEditorFactory propertyEditorFactory = EDITORS.get(paramType);
            if (propertyEditorFactory != null) {
                return propertyEditorFactory;
            }
            paramType = paramType.getSuperclass();
        }
        return null;
    }


    public static String getterPropName(Method method) {
        String name;
        if (method.getName().startsWith("is")) {
            name = method.getName().substring("is".length());
        } else if (method.getName().startsWith("get")) {
            name = method.getName().substring("get".length());
        } else {
            throw new IllegalStateException();
        }
        return name.substring(0, 1).toLowerCase() + name.substring(1);
    }

    public static String setterPropName(Method method) {
        check(method.getName().startsWith("set"), IllegalStateException::new);
        String name = method.getName().substring("set".length());
        return name.substring(0, 1).toLowerCase() + name.substring(1);
    }

}
