package com.steammachine.propertyeditor.baseeditors;

import static com.steammachine.propertyeditor.EditorsUtils.getterPropName;
import static com.steammachine.propertyeditor.EditorsUtils.setterPropName;
import static com.steammachine.propertyeditor.types.EditorTags.Tag.READ_ONLY;
import static com.steammachine.propertyeditor.types.EditorTags.Tag.VALUE_NULLABLE;
import static com.steammachine.propertyeditor.utils.CommonUtils.check;
import static com.steammachine.propertyeditor.utils.CommonUtils.wrapFunc;
import static com.steammachine.propertyeditor.utils.CommonUtils.wrapProc;
import static com.steammachine.propertyeditor.utils.EditorTagsUtils.propertyEditorTags;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.StreamSupport.stream;

import com.steammachine.propertyeditor.EditorsUtils;
import com.steammachine.propertyeditor.types.EditorTags.Tag;
import com.steammachine.propertyeditor.types.PropertyEditor;
import com.steammachine.propertyeditor.types.Selection;
import com.steammachine.propertyeditor.types.Value;
import com.steammachine.propertyeditor.types.exceptions.NullValueIsNotAllowed;
import com.steammachine.propertyeditor.types.exceptions.UnableToSetReadonlyProperty;
import com.steammachine.propertyeditor.types.exceptions.UnableToSetValue;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;
import lombok.AllArgsConstructor;
import lombok.Data;

public abstract class DefaultPropertyEditor<T> implements PropertyEditor<T> {

    @Data
    @AllArgsConstructor
    private static class ValueItem<T> implements Value<T> {

        private final Object nexus;
        private final Method getter;
        private final Method setter;

        @Override
        public void set(T value) {
            wrapProc(() -> getSetter().invoke(nexus, value),
                    thr -> new IllegalStateException());
        }

        @Override
        public T get() {
            //noinspection unchecked
            return wrapFunc(() -> (T) getGetter().invoke(nexus), IllegalStateException::new);
        }

        public boolean isReadOnly() {
            return setter == null;
        }

    }

    private final String name;
    private final Selection selection;
    private final Class<T> type;
    private final List<Value<T>> dataList;
    private final Set<Tag> editorTags = propertyEditorTags(getClass());


    DefaultPropertyEditor(String name, Class<T> type, Selection selection) {
        this.name = requireNonNull(name);
        this.type = requireNonNull(type);
        this.selection = requireNonNull(selection);
        this.dataList = stream(selection.spliterator(), false).map(
                o -> {
                    Method getter = Arrays.stream(o.getClass().getMethods())
                            .filter(method -> method.getReturnType() == type)
                            .filter(EditorsUtils::isGetter)
                            .filter(method -> getterPropName(method).equals(name))
                            .findFirst().orElseThrow(IllegalStateException::new);
                    Method setter = Arrays.stream(o.getClass().getMethods())
                            .filter(EditorsUtils::isSetter)
                            .filter(method -> setterPropName(method).equals(name))
                            .filter(method -> method.getParameterCount() == 1)
                            .filter(method -> method.getParameterTypes()[0] == type)
                            .findFirst().orElse(null);

                    return new ValueItem<T>(o, getter, setter);
                }).collect(toList());
    }

    @Override
    public void setValue(T value) {
        check(!isReadOnly(), UnableToSetReadonlyProperty::new);
        check(value != null || hasTag(VALUE_NULLABLE), NullValueIsNotAllowed::new);
        dataList.forEach(item -> wrapProc(() -> item.set(value), UnableToSetValue::new));
    }

    public boolean isReadOnly() {
        return hasTag(READ_ONLY) || dataList.get(0).isReadOnly();
    }



    private boolean hasTag(Tag tag) {
        return editorTags.contains(tag);
    }

    protected int propertyCount() {
        return dataList.size();
    }

    public T getValueAt(int index) {
        return dataList.get(index).get();
    }

    public String getName() {
        return name;
    }

    public Selection getSelection() {
        return selection;
    }

    public Class<T> propertyType() {
        return type;
    }

    public boolean allEquals() {
        if (propertyCount() == 1) {
            return true;
        }
        T first = getValueAt(0);
        return Stream.iterate(1, i -> i + 1).limit((long) (propertyCount()) - 1)
                .map(this::getValueAt).allMatch(t -> Objects.equals(first, t));
    }

    @Override
    public String propertyName() {
        return name;
    }

}
