(ns taoensso.tukey.rss
  "RollingSummaryStats.
  Private ns, implementation detail."
  {:author "Peter Taoussanis (@ptaoussanis)"}
  (:require
   [taoensso.encore :as enc :refer [have have? have!]]
   [taoensso.tukey.impl :as impl])

  #?(:clj (:import [java.util LinkedList])))

;;;;

(defn- acc-new [init]
  #?(:clj  (if init (LinkedList.     init) (LinkedList.))
     :cljs (if init (cljs.core/array init) (cljs.core/array))))

(defn- acc-add [acc x]
  #?(:clj  (.add ^LinkedList acc x)
     :cljs (.push            acc x)))

(defn- acc-len ^long [acc]
  #?(:clj  (.size ^LinkedList acc)
     :cljs (alength           acc)))

(defprotocol IRollingSummaryStats
  (^:private rss-get   [_])
  (^:private rss-merge [_])
  (^:private rss-add   [_ n]))

;;;;

(deftype RollingSummaryStats [msstats_ acc_ max-acc-len merge-counter]
  Object
  (toString [_] ; "RollingSummaryStats[n=1, pending=8, merged=0]"
    (str
      "RollingSummaryStats[n=" (get @msstats_ :n 0) ", "
      "pending=" (acc-len @acc_)                    ", "
      "merged="  @merge-counter                     "]"))

  #?@(:clj  [clojure.lang.IDeref ( deref [this] (rss-get this))]
      :cljs [             IDeref (-deref [this] (rss-get this))])

  #?@(:clj  [clojure.lang.IFn ( invoke [this n] (rss-add this n))]
      :cljs [             IFn (-invoke [this n] (rss-add this n))])

  IRollingSummaryStats
  (rss-get [this] (deref (or (rss-merge this) @msstats_)))
  (rss-merge [_]
    (let [[drained] (reset-vals! acc_ (acc-new nil))]
      (if (== (acc-len drained) 0)
        nil
        (let [msstats-drained (impl/summary-stats drained)
              msstats-merged  (swap! msstats_ impl/summary-stats-merge
                                msstats-drained)]
          (merge-counter)
          msstats-merged))))

  (rss-add [this n]
    (let [acc @acc_]
      (acc-add acc n)

      (when-let [^long nmax max-acc-len]
        (when (> (acc-len acc) nmax)
          (rss-merge this)))

      nil)))

(defn ^:public rolling-summary-stats
  "EXPERIMENTAL, SUBJECT TO CHANGE!
  Returns a stateful RollingSummaryStats so that:
    (deref rss) => {:keys [n min max p25 ... p99 mean var mad]}
    (rss n)     => Adds given num to accumulator."

  ([] (rolling-summary-stats nil))
  ([{:keys [max-acc-len init-sstats init-acc]
     :or   {max-acc-len 1e5}}]

   (RollingSummaryStats.
     (atom init-sstats)
     (atom (acc-new init-acc))
     (long max-acc-len)
     (enc/counter))))

(comment
  (let [rss (rolling-summary-stats {:max-acc-len 10})]
    [(dotimes [_ 1e5] (rss (rand-int 1000))) (str rss) @rss]))

;;;; Print methods

#?(:clj
   (let [ns *ns*]
     (defmethod print-method RollingSummaryStats
       [x ^java.io.Writer w] (.write w (str "#" ns "." x)))))
