(ns teachscape.etlio.server.scheduler
  "Manages tasks and their scheduled execution"
  (:refer-clojure :exclude [remove reset!])
  (:require [clojurewerkz.quartzite.scheduler :as qs]
            [taoensso.timbre                  :as log]
            [teachscape.etlio.evaluator       :as ev]
            [teachscape.etlio.server.config   :as cfg]
            [clojurewerkz.quartzite.jobs       :as qj]
            [clojurewerkz.quartzite.triggers   :as qt]
            [clojurewerkz.quartzite.conversion :as qc]
            [teachscape.etlio.server.fs :refer [path->str]]
            [clojure.stacktrace :as st]
            [me.raynes.fs       :as fs])
  (:import java.nio.file.Path))

(def ^:const group-name "tasks")

(def known-files  (ref {}))
(def known-tasks  (ref {}))
(def name-to-path (ref {}))

(defn ignore?
  [m]
  (nil? (:schedule m)))

(qj/defjob ExecuteTask
  [ctx]
  (let [m (qc/from-job-data ctx)]
    (if-let [task (get @known-tasks (get m "task-name"))]
      (do
        (log/info (format "Running task %s" (:name task)))
        (try
          (ev/run-task task cfg/config)
          (catch Exception e
            (log/error (format "Caught an exception when running task %s" (:name task)))
            (st/print-cause-trace e))))
      (log/warn (format "Can't find task %s among active (job data: %s)" (get m "task-name") m)))))

(defn key-name-for
  "Returns string key for given task name"
  [^String name]
  (format "tasks.%s" name))

(defn trigger-key-for
  "Returns trigger key for given task"
  [{:keys [name] :as task}]
  (qt/key group-name (key-name-for name)))

(defn job-for
  [{:keys [name] :as task}]
  (log/debug "Task name is " name)
  (qj/build
   (qj/of-type ExecuteTask)
   (qj/using-job-data {"task-name" name})
   (qj/with-identity group-name (key-name-for name))))

(defn trigger-for
  [{:keys [name schedule]}]
  (qt/build
   (qt/with-identity (trigger-key-for {:name name}))
   (qt/start-now)
   (qt/with-schedule schedule)))

(defn task-by-name
  [^String name]
  (get @known-tasks name))

(defn task-by-path
  [path]
  (get @known-files path))

(defn register-task
  [^String task-name task-path task]
  (dosync
   (commute known-tasks assoc task-name task)
   (commute known-files assoc task-path task)
   (commute name-to-path assoc task-name task-path)))

(defn deregister-task
  [^String task-name task-path]
  (dosync
   (let [task (task-by-name task-name)]
     (commute known-tasks dissoc task-name)
     (commute known-files dissoc task-path)
     (commute name-to-path dissoc task-name))))

(defn deregister-all
  []
  (dosync
   (ref-set known-tasks  {})
   (ref-set known-files  {})
   (ref-set name-to-path {})))

(defn schedule
  "Adds a task to the scheduler"
  [^String task-name task-path task]
  (let [job (job-for task)
        trg (trigger-for task)]
    (qs/maybe-schedule job trg)
    task))

(defn unschedule
  "Removes a task from the scheduler"
  [^String task-name task-path task]
  (let [trg-key (trigger-key-for task)]
    ;; if the job is not durable, this will remove the
    ;; job automatically
    (qs/delete-trigger trg-key)))

(defn reschedule
  "Unschedules and reschedules a task"
  [^String task-name task-path task]
  (unschedule task-name task-path task)
  (schedule task-name task-path task))

;;
;; API
;;

(defn run
  "Runs the scheduler"
  []
  (qs/initialize)
  (qs/start))

(defn add!
  "Adds a task to the scheduler. Will use the task's schedule to execute it
   periodically."
  [^Path task-path]
  (log/debug (format "About to compile task in %s" (.toFile task-path)))
  (let [task      (ev/read-task (path->str task-path))
        task-name (:name task)]
    (if (ignore? task)
      (log/warn (format "Task %s has no schedule, ignoring" task-name))
      (do
        (register-task task-name task-path task)
        (schedule task-name task-path task)
        (log/info (format "Added task %s to the scheduler" task-name))))))

(defn remove!
  "Removes a task to the scheduler. The task will not be executed again."
  [^Path task-path]
  (if-let [task (task-by-path task-path)]
    (let [task-name (:name task)]
      (unschedule task-name task-path task)
      (deregister-task task-name task-path)
      (log/info (format "Removed task %s from the scheduler" task-name)))
    (log/warn (format "Could not find a task for %s!" task-path))))

(defn remove-by-name!
  "Removes a task to the scheduler by name. The task will not be executed again."
  [^String name]
  (if-let [^Path path (get @name-to-path name)]
    (do
      (remove! path)
      true)
    (do
      (log/warn (format "Task %s is not known to the scheduler!" name))
      false)))

(defn reload!
  "Reloads a task in the scheduler. Will use the updated task's schedule to execute it
   periodically."
  [^Path task-path]
  (log/debug (format "About to [re]compile task in %s" (.toFile task-path)))
  (let [task      (ev/read-task (path->str task-path))
        task-name (:name task)]
    (register-task task-name task-path task)
    ;; the schedule may have changed, re-schedule
    (reschedule task-name task-path task)
    (log/info (format "Updated task %s in the scheduler" task-name))))

(defn preload
  "Preloads tasks in the given directory"
  [^Path dir]
  (let [filenames (fs/list-dir (.toFile dir))
        paths     (map (fn [^String filename]
                         (.resolve dir filename))
                       filenames)]
    (doseq [^Path p paths]
      (try
        (log/info (format "Loading a task from %s" (str p)))
        (reload! p)
        (catch Exception e
          (log/error (format "Could not load the task at %s due to exception: %s" (str p) e)))))))

(defn reset!
  "Resets the schduler by removing all tasks from it.
   Optionally waits for currently running tasks to complete."
  ([]
     (reset! false))
  ([wait-for-completion?]
     (qs/shutdown wait-for-completion?)
     (qs/recreate)
     (deregister-all)
     true))

(defn known-path?
  [^Path path]
  (not (nil? (get @known-files path))))
