(ns teachscape.etlio.tools.server
  "Daemon tool that executes and manages tasks"
  (:require [clojure.tools.cli :refer [cli]]
            [clojure.java.io                   :as io]
            [clojure.edn                       :as edn]
            [teachscape.etlio.evaluator        :as ev]
            [teachscape.etlio.server.watchers  :as ew]
            [teachscape.etlio.server.scheduler :as es]
            [teachscape.etlio.server.http      :as ews]
            [teachscape.etlio.server.config    :as cfg]
            [taoensso.timbre                   :as log]
            [me.raynes.fs                      :as fs]
            [teachscape.etlio.server.fs        :as sfs]
            [teachscape.etlio.tools.cli        :as ecl]
            nio2.io)
  (:import java.nio.file.Path
           java.io.File)
  (:gen-class))


;;
;; Implementation
;;

(defn handle-created-file
  [^Path path]
  (log/info (format "%s was created" path))
  (es/add! path))

(defn handle-modified-file
  [^Path path]
  (log/info (format "%s was modified" path))
  (es/reload! path))

(defn handle-deleted-file
  [^Path path]
  (log/info (format "%s was deleted" path))
  (es/remove! path))

(defn handle-watch-event
  "Handles filesystem events in the tasks directory"
  [{:keys [kind ^Path path]}]
  (let [f (case kind
            :create handle-created-file
            :delete handle-deleted-file
            :modify handle-modified-file)]
    (try
      (f (sfs/path-for-task path))
      (catch Exception e
        (log/error (format "Failed to handle a watch event on %s" path))))))

(defn start-dir-watcher
  [tasks-dir]
  (log/info "Starting directory watcher")
  (ew/watch-dir tasks-dir handle-watch-event))

(defn start-http-server
  [port]
  (log/debug (format "About to start HTTP server on port %d" port))
  (try
    (ews/start-server port)
    (catch Exception e
      (log/error (format "Caught an exception when trying to start HTTP server: %s" e))))
  (log/info (format "Started HTTP server on port %d" port)))

(defn process-args
  [{:keys [^String tasks-dir config-file http-server http-port] :as opts} args banner]
  (when (not (ecl/valid-tasks-directory? tasks-dir))
    (ecl/exit! 1 (format "%s does not exist or is not a directory (when checking tasks-dir)" tasks-dir)))
  (when (and config-file (not (ecl/valid-config-file? config-file)))
    (ecl/exit! 1 (format "%s does not exist or is not a readable Clojure file (when checking config-file)" config-file)))
  (let [ctx  (edn/read-string (slurp (io/reader config-file)))
        path (-> (File. tasks-dir) .toPath .toAbsolutePath)]
    (cfg/set-config! ctx)
    (es/run)
    (log/info "Started Quartz scheduler")
    (sfs/use-dir! path)
    (es/preload path)
    (if http-server
      (start-http-server http-port)
      (log/info "Not starting HTTP server"))
    (log/info (format "Will watch %s for changes" tasks-dir))
    (start-dir-watcher tasks-dir)))

(defn entry-point
  [args]
  (let [[opts args banner] (cli args
                                ["-h" "--help"             "Show help" :flag true :default false]
                                ["-T" "--tasks-dir"        "REQUIRED: Path to directory with task definition files"]
                                ["-C" "--config-file"      "Path to configuration file"]
                                ["-S" "--[no-]http-server" "Start HTTP server?" :default true]
                                ["-p" "--http-port"        "HTTP server port" :parse-fn #(Long/valueOf ^String %) :default 17771])]
    (when (:help opts)
      (println banner)
      (ecl/exit! 0))
    (process-args opts args banner)))


;;
;; API
;;

(defn -main
  [& args]
  (entry-point args))
