(ns teachscape.etlio.evaluator
  "Task compilation and evaluation implementation"
  (:refer-clojure :exclude [name])
  ;; force compilation
  (:require teachscape.etlio.dsl
            [metrics.counters   :as cm]
            [metrics.histograms :as hm]
            [taoensso.timbre    :as log]
            [me.raynes.fs       :as fs]
            [clojure.java.io    :as io]
            [teachscape.etlio.time :refer [time-body]]
            [clojure.set        :as set]
            [clojurewerkz.cyclist.dependencies :as cy])
  (:import java.io.File))

;;
;; Metrics
;;

(def tasks-executed
  "Counts total number of executed tasks"
  (cm/counter ["etlio" "evaluator" "tasks-executed"]))
(def execution-time
  "Histogram of task execution time, in nanoseconds"
  (hm/histogram ["etlio" "evaluator" "execution-time"]))

;;
;; Implementation
;;

(defn ^:private perform-setup
  [{:keys [setup context] :or {context {} setup []}} ctx]
  (let [merged-ctx (merge ctx context)]
    (doseq [f setup]
      (f merged-ctx))))

(defn ^:private perform-finish
  [{:keys [finish context] :or {context {} finish []}} ctx stats]
  (let [merged-ctx (merge ctx context)]
    (doseq [f finish]
      (f merged-ctx))))

(defn ^:private load-inputs
  [{:keys [inputs context name] :or {inputs [] context {}}} ctx]
  (log/debug (format "loading inputs for task %s" name))
  (let [merged-ctx (merge ctx context)]
    (when-not (seq inputs)
      (throw (IllegalArgumentException. (format "Task %s has no inputs" name))))
    (mapcat (fn [f]
              (f merged-ctx))
            inputs)))

(defn ^:private transform-inputs
  [{:keys [transforms name] :or {transforms []}} rows]
  (log/debug (format "transforming inputs for task %s" name))
  (reduce (fn [acc f]
            (map f acc))
          rows
          transforms))

(defn ^:private load-outputs
  [{:keys [outputs name context] :or {outputs []}} rows ctx]
  (log/debug (format "loading outputs for task %s" name))
  (let [merged-ctx (merge ctx context)]
    (when-not (seq outputs)
      (throw (IllegalArgumentException. (format "Task %s has no outputs" name))))
    (reduce (fn [n _] (inc n)) 
            0
            (doseq [f outputs]
              (f merged-ctx rows)))))
  
  (defn ^:private find-dependency
    [dep-name deps]
    {:post [(not (nil? %))]}
    (some (fn [{:keys [name] :as m}]
            (when (= (clojure.core/name name) (clojure.core/name dep-name))
              m))
          deps))
  
  (defn ^:private files-in
    "Returns a sequence of files in under the given directory"
    [^String dir]
    (filter (memfn ^java.io.File isFile)
            (.listFiles (io/file dir))))
  
  (defn check-unknown-tasks
    "Throws IllegalArgumentException if provided tasks have references
    to unknown tasks"
    [ms]
    (let [names (set (map (comp clojure.core/name :name)
                          ms))
          deps  (set (map clojure.core/name
                          (flatten (mapcat :dependencies ms))))]
      (when-not (set/subset? deps names)
        (throw (IllegalArgumentException. (format "Some tasks listed in dependencies are not found in the tasks directory: %s" (set/difference deps names)))))))
  
  (defn check-cyclic-references
    "Throws IllegalArgumentException if provided tasks have cyclic dependencies"
    [ms]
    (let [deps (set (map (fn [{:keys [name dependencies]}]
                           {:name (clojure.core/name name)
                            :dependencies (set (map clojure.core/name (flatten dependencies)))})
                         ms))]
      (when-let [d (cy/detect deps)]
        (throw (IllegalArgumentException. "Detected cyclic dependencies between tasks")))))
  
  ;;
  ;; API
  ;;
  
  (defn execute-task
    [m ctx]
    (let [xs         (load-inputs m ctx)
          ys         (transform-inputs m xs)]
      (load-outputs m ys ctx)))
  
  (defn execute-with-setup
    [execution m ctx]
    (perform-setup m ctx)
    (log/debug (format "executing with %s" execution))
    (execution m ctx))
  
  (defn read-task
    "Reads a task from a file, evaluates it and returns as a map"
    [^String path]
    (if-let [eval-ctx (find-ns 'teachscape.etlio.dsl)]
      (binding [*ns* eval-ctx]
        (try (load-file path)
          (catch Exception e
            (throw (Exception. (format "Error loading %s" path) e)))))
      (throw (IllegalStateException. "Could not find DSL ns to evaluate in!"))))
  
  (defn read-tasks
    "Reads all tasks from a directory, evaluates them and returns
    as a collection of maps"
    [^String path]
    (when-not (and (fs/exists? path)
                   (fs/directory? path))
      (throw (IllegalArgumentException. (format "%s does not exist or is not a directory" path))))
    (let [files (map str (files-in (io/file path)))]
      (doall (map read-task files))))
  
  (defn read-tasks-as-map
    "Reads all tasks from a directory, evaluates them and returns
    as a map from paths to task maps"
    [^String path]
    (when-not (and (fs/exists? path)
                   (fs/directory? path))
      (throw (IllegalArgumentException. (format "%s does not exist or is not a directory" path))))
    (let [files (map str (files-in (io/file path)))]
      (reduce (fn [acc ^String s]
                (assoc acc (-> (File. s) .toPath .toAbsolutePath) (read-task s)))
              {}
              files)))
  
  (defn validate-tasks
    "Validates tasks: detects cyclic dependencies, references
    to unknown tasks and so on"
    [ms]
    (check-unknown-tasks ms)
    (check-cyclic-references ms))
  
  (defn run-task
    "Executes a task without executing its dependencies"
    ([m]
     (run-task m {}))
    ([m ctx]
     (let [execution (if (nil? (:execution m)) execute-task (:execution m))
           start  (System/nanoTime)
           rows-from-output (execute-with-setup execution m ctx)
           nanos (- (System/nanoTime) start)
           stats (when rows-from-output {:execution-time nanos :rows-handled rows-from-output})]
       (perform-finish m ctx stats)
       (hm/update! execution-time nanos)
       (cm/inc! tasks-executed)
       true)))
  
  (defn run-task-and-dependencies
    "Executes a task and its dependencies"
    ([m ctx deps]
     (run-task-and-dependencies m ctx deps nil))
    ([m ctx deps parent-task]
     (if parent-task
       (log/info (format "Executing task %s and its dependencies as a dependency of %s" (:name m) parent-task))
       (log/info (format "Executing task %s and its dependencies" (:name m))))
     (let [execution (if (nil? (:execution m)) execute-task (:execution m))
           start  (System/nanoTime)
           rows-from-output (execute-with-setup execution m ctx)
           nanos (- (System/nanoTime) start)
           stats (when rows-from-output {:execution-time nanos :rows-handled rows-from-output})]
       (perform-finish m ctx stats)
       (hm/update! execution-time nanos)
       (cm/inc! tasks-executed)
       (when-let [dep-names (seq (:dependencies m))]
         (doseq [nm dep-names]
           (run-task-and-dependencies (find-dependency nm deps) ctx deps (:name m))))
       true)))
