(ns com.timezynk.mongo.schema
  (:require
   [clojure.tools.logging :as log]))

(def ^:const ID     "objectId")
(def ^:const STRING "string")

(def types [ID STRING])

(defn- ^:no-doc set-required [{:keys [optional?]}]
  {:required (not optional?)})

(defn id
  "Schema type `ObjectId`.
   
   | Parameter    | Description |
   | ---          | --- |
   | `:optional?` | `optional boolean` Is the field optional? |

   **Examples**
   
   ```Clojure
   (create-collection! :users :schema {:id (id :optional? true)})
   ```"
  {:arglists '([& :optional? <boolean>])}
  [& options]
  (merge {:bsonType ID}
         (set-required options)))

(defn string
  "Schema type `string`.
   
   | Parameter    | Description |
   | ---          | --- |
   | `:optional?` | `optional boolean` Is the field optional? |

   **Examples**
   
   ```Clojure
   (create-collection! :users :schema {:name (string :optional? true)})
   ```"
  [& options]
  (merge {:bsonType STRING}
         (set-required options)))

(defn ^:no-doc convert-schema [schema]
  (let [required (reduce-kv (fn [l k v]
                              (if (:required v)
                                (cons (name k) l)
                                l))
                            [] schema)
        schema   (->> schema
                      (map (fn [[k v]]
                             [k (dissoc v :required)]))
                      (into {}))
        properties (merge {:_id  {:bsonType ID}}
                          schema)]
    {:$jsonSchema {:bsonType "object"
                   :required required
                   :properties properties
                   :additionalProperties false}}))