(ns ^:no-doc com.timezynk.mongo.methods.watch
  (:require
   [clojure.core.async :as async]
   [clojure.tools.logging :as log]
   [com.timezynk.mongo.config :refer [*mongo-session*]]
   [com.timezynk.mongo.utils.collection :as coll]
   [com.timezynk.mongo.utils.convert :as convert])
  (:import [com.mongodb.client MongoCollection MongoCursor MongoIterable]
           [com.mongodb.client.model.changestream ChangeStreamDocument UpdateDescription]
           [com.mongodb MongoException]
           [java.util Date]
           [org.bson BsonDocument BsonTimestamp Document]))

(defmulti watch-method ^MongoIterable
  (fn [^MongoCollection _coll ^String _op-type]
    {:session (some? *mongo-session*)}))

(defmethod watch-method {:session true} [coll op-type]
  (.watch coll *mongo-session* [(convert/clj->doc {:$match {:operationType op-type}})]))

(defmethod watch-method {:session false} [coll op-type]
  (.watch coll [(convert/clj->doc {:$match {:operationType op-type}})]))

(defn- get-cursor [coll op-type]
  (-> ^MongoCollection (coll/get-collection coll)
      ^MongoIterable (watch-method op-type)
      ^MongoCursor (.iterator)))

(defn- get-time [event]
  (-> ^BsonTimestamp (.getClusterTime event)
      ^Long (.getTime)
      (* 1000)
      (Date.)))

(defn insert-method [coll insert-fn]
  (let [cursor (get-cursor coll "insert")]
    (async/thread
      (try
        (while (.hasNext cursor)
          (let [event ^ChangeStreamDocument (.next cursor)
                time  (get-time event)
                doc   (-> ^Document (.getFullDocument event)
                          (convert/doc->clj))]
            (insert-fn time doc)))
        (.close cursor)
        (catch MongoException e
          (if (= (.getMessage e) "state should be: open")
            (log/info "Cursor closed, insert watch terminated")
            (throw (MongoException. e))))))))

(defn update-method [coll update-fn]
  (let [cursor (get-cursor coll "update")]
    (async/thread
      (try
        (while (.hasNext cursor)
          (let [event  ^ChangeStreamDocument (.next cursor)
                time   (get-time event)
                id     (-> ^BsonDocument (.getDocumentKey event)
                           (Document.)
                           (convert/doc->clj))
                update (-> ^UpdateDescription (.getUpdateDescription event)
                           ^BsonDocument (.getUpdatedFields)
                           (Document.)
                           (convert/doc->clj))]
            (update-fn time (merge id update))))
        (.close cursor)
        (catch MongoException e
          (if (= (.getMessage e) "state should be: open")
            (log/info "Cursor closed, update watch terminated")
            (throw (MongoException. e))))))))