(ns ^:no-doc com.timezynk.mongo.helpers
  (:require
   [com.timezynk.mongo.codecs.bson :refer [->bson]]
   [com.timezynk.mongo.codecs.collection :refer [collection-provider]]
   [com.timezynk.mongo.codecs.map :refer [map-provider]]
   [com.timezynk.mongo.config :refer [*mongo-database*]]
   [com.timezynk.mongo.convert :refer [it->clj]]
   [com.timezynk.mongo.guards :refer [*insert-guard* *update-guard* catch-return]]
   [com.timezynk.mongo.methods.fetch :refer [fetch-method]]
   [com.timezynk.mongo.methods.fetch-and-update :refer [fetch-and-update-method fetch-and-update-options]]
   [com.timezynk.mongo.methods.insert :refer [insert-method insert-options]]
   [com.timezynk.mongo.methods.update :refer [update-method update-one-method update-options]])
  (:import [com.mongodb.client MongoCollection]
           [org.bson.codecs.configuration CodecRegistries]))

(defn codec-registry [database codecs]
  (.withCodecRegistry database
                      (CodecRegistries/fromRegistries
                       [(CodecRegistries/fromProviders [(collection-provider)
                                                        (map-provider)])
                        (CodecRegistries/fromCodecs codecs)])))

(defmacro get-collection ^MongoCollection [coll]
  `(.getCollection *mongo-database*
                   (name ~coll)
                   clojure.lang.PersistentArrayMap))

(defmacro do-fetch [coll query options]
  `{:pre [~coll ~query]}
  `(-> (fetch-method (get-collection ~coll)
                     (->bson ~query)
                     ~options)
       (it->clj)))

(defmacro do-insert [coll docs options]
  `{:pre [~coll]}
  `(catch-return
    (*insert-guard* ~docs)
    (-> (get-collection ~coll)
        (insert-options ~options)
        (insert-method ~docs))))

(defmacro do-update [coll query update options]
  `{:pre [~coll ~query]}
  `(catch-return
    (*update-guard* ~update)
    (let [result# (update-method (get-collection ~coll)
                                 (->bson ~query)
                                 (->bson ~update)
                                 (update-options ~options))]
      {:matched-count  (.getMatchedCount result#)
       :modified-count (.getModifiedCount result#)})))

(defmacro do-update-one [coll query update options]
  `{:pre [~coll ~query]}
  `(catch-return
    (*update-guard* ~update)
    (let [result# (update-one-method (get-collection ~coll)
                                     (->bson ~query)
                                     (->bson ~update)
                                     (update-options ~options))]
      {:matched-count  (.getMatchedCount result#)
       :modified-count (.getModifiedCount result#)})))

(defmacro do-fetch-and-update-one [coll query update options]
  `{:pre [~coll ~query]}
  `(catch-return
    (*update-guard* ~update)
    (-> (fetch-and-update-method (get-collection ~coll)
                                 (->bson ~query)
                                 (->bson ~update)
                                 (fetch-and-update-options ~options)))))
