(ns com.timezynk.useful.mongo.queue
  (:require
   [clojure.tools.logging :as log]
   [com.timezynk.useful.prometheus.core :as metrics]
   [com.timezynk.useful.timed-queue :refer [TimedQueue]]
   [somnium.congomongo :as mongo]))

(def ^:const DEFAULT_MIN_INTERVAL 250)
(def ^:const DEFAULT_MIN_SLEEP 100)

(defprotocol WorkQueue
  (start-worker! [this thread-name] "Start queue worker")
  (stop-worker! [this] "Stop queue worker"))

(defn- fetch-next-item! [collection ts]
  (mongo/fetch-and-modify collection
                          {:run-at {:$lte ts}}
                          nil
                          :remove? true
                          :sort {:run-at 1}))

(defn- process-queue! [collection handler message-counter user-time]
  (when-let [item (fetch-next-item! collection (System/currentTimeMillis))]
    (let [start-time (System/nanoTime)
          type (:type item)]
      (log/debug "Queue item of type" type "in" collection "is ready to run. Latency" (- (System/currentTimeMillis) (:run-at item)) "ms")
      (handler item)
      (metrics/inc! message-counter type)
      (metrics/inc-by! user-time (/ (double (- (System/nanoTime) start-time)) 1000000000.0) type))))

(defn- add-payload-prefix [acc [k v]]
  (assoc acc (str "payload." (name k)) v))

(deftype MongoQueue
         [collection handler min-interval min-sleep go-on? worker-thread user-time message-counter]

  WorkQueue
  (start-worker! [this thread-name]
    (reset! go-on? true)
    (reset! worker-thread
            (Thread.
             (bound-fn []
               (log/info "Starting worker for queue" collection)
               (while @go-on?
                 (try
                   (let [next-ts (+ (System/currentTimeMillis) min-interval)]
                     (process-queue! collection handler message-counter user-time)
                     (let [sleep (max min-sleep (- next-ts (System/currentTimeMillis)))]
                       (Thread/sleep sleep)))
                   (catch java.lang.InterruptedException e
                     (log/info "Queue" collection "closing"))
                   (catch Exception e
                     (log/warn e "Exception in queue" collection))))
               (log/info "Queue" collection "finished"))
             thread-name))
    (doto @worker-thread
      (.setDaemon false)
      (.setPriority Thread/MIN_PRIORITY)
      (.start)))

  (stop-worker! [this]
    (reset! go-on? false)
    (when @worker-thread
      (.interrupt @worker-thread)
      (reset! worker-thread nil)))

  TimedQueue
  (push-job! [this type run-at payload]
    (mongo/insert! collection
                   {:run-at run-at
                    :type type
                    :payload payload}
                   :write-concern :unacknowledged))

  (upsert-job! [this type run-at selector update]
    (mongo/update! collection
                   (merge
                    {:type type}
                    (reduce add-payload-prefix {} selector))
                   (merge
                    {:$set
                     {:run-at run-at
                      :type type}}
                    update)
                   :upsert true
                   :write-concern :unacknowledged)))

(defn create-mongo-queue
  ([id collection handler]
   (create-mongo-queue id collection handler DEFAULT_MIN_INTERVAL DEFAULT_MIN_SLEEP))
  ([id collection handler min-interval min-sleep]
   (MongoQueue.
    collection handler min-interval min-sleep
    (atom true)
    (atom nil)
    (metrics/counter
     (keyword (str (name id) "_user_time_seconds"))
     (str "A counter of the total user time used for " id)
     :type)
    (metrics/counter
     (keyword (str (name id) "_messages_total"))
     (str "A counter of the total number of messages processed in " id)
     :type))))

