(ns com.timezynk.useful.prometheus.middleware
  (:require
   [clojure.string :as string]
   [clojure.tools.logging :as log]
   [clojure.string :refer [replace]]
   [com.timezynk.useful.prometheus.core :as metrics]
   [clojure.test :refer [deftest is testing]]
   [com.timezynk.useful.date :refer [iso-pattern]])
  (:import [java.lang.management ManagementFactory]))

(def ^:const UA_PRODUCTS #"[^ ()/]+/[^ ()/]+")
(def ^:const SLOW_LOG_LIMIT 0.5)

(defonce request-counter (metrics/counter :http_requests_total
                                          "A counter of the total number of HTTP requests processed"
                                          :method :status :user_agent :version))

(defonce request-seconds (metrics/counter :http_requests_seconds
                                          "A counter of the total seconds spent processing HTTP requests"
                                          :method :uri))

(defonce histogram (metrics/histogram :http_request_latency_seconds
                                      "A histogram of the response latency for HTTP requests in seconds."
                                      [0.001, 0.005, 0.010, 0.020, 0.050, 0.100, 0.200, 0.300, 0.500, 0.750, 1, 2, 4, 8]
                                      :method :status_class))

(defn get-products [user-agent]
  (clojure.string/join " "
                       (re-seq UA_PRODUCTS
                               (clojure.string/replace user-agent "Mozilla/5.0" ""))))

(defn get-client [{:keys [headers query-params]}]
  (let [user-agent (or (get headers "user-agent") "")]
    (get-products user-agent)))

(defn get-uri [{:keys [uri]}]
  (-> uri
      (replace #"[0-9a-f]{24}" ":oid")
      (replace #"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(\.\d{3})?" ":datetime")
      (replace iso-pattern ":date")))

(def thread-bean (delay (ManagementFactory/getThreadMXBean)))

(defn wrap-metrics [handler version]
  (fn [request]
    (let [request-method (:request-method request)
          start-cpu-time (.getCurrentThreadCpuTime @thread-bean)
          start-time (System/currentTimeMillis)
          response (handler request)
          finish-time (System/currentTimeMillis)
          finish-cpu-time (.getCurrentThreadCpuTime @thread-bean)
          cpu-time-diff (/ (double (- finish-cpu-time start-cpu-time)) 1000000000.0)
          response-status (get response :status 404)
          request-time (/ (double (- finish-time start-time)) 1000.0)
          status-class (str (int (/ response-status 100)) "XX")
          method-label (string/upper-case (name request-method))
          client (get-client request)
          uri (get-uri request)]
      (apply metrics/observe! histogram request-time [method-label status-class])
      (apply metrics/inc! request-counter [method-label (str response-status) client version])
      (apply metrics/inc-by! request-seconds cpu-time-diff [method-label uri])
      (when (> request-time SLOW_LOG_LIMIT)
        (log/warn "Slow request" request-time "seconds:" (.toUpperCase (name request-method)) (:uri request)))
      response)))

(deftest anonymize-uri
  (testing "replacing object id"
    (is (= "/api/flexitime/:oid" (get-uri {:uri "/api/flexitime/617a641e364296cd6f83f7c8"})))
    (is (= "/api/shifts/:oid/booked-users" (get-uri {:uri "/api/shifts/617a6a6557261bb052a00fda/booked-users"}))))
  (testing "replacing dates"
    (is (= "/api/flexitime/:oid/diff/:date/" (get-uri {:uri "/api/flexitime/617a641e364296cd6f83f7c8/diff/2021-09-10/"}))))
  (testing "replacing datetime"
    (is (= "/api/wat/:datetime/summary" (get-uri {:uri "/api/wat/2021-09-10T08:29:30/summary"})))))
