(ns com.timezynk.useful.date
  (:refer-clojure :exclude [format])
  (:require
   [com.timezynk.useful.rest.current-user :as current-session]
   [somnium.congomongo.coerce :as cc]
   [tzbackend.util.constants :refer [DEFAULT_TIMEZONE]])
  (:import [java.time Duration Instant LocalDate LocalDateTime LocalTime YearMonth ZonedDateTime ZoneId ZoneOffset]
           [java.time.format DateTimeFormatter DateTimeParseException]))

(defn get-zone
  "Return the timezone object for the current session,
   or the default timezone object if the former can't be found."
  ^ZoneId []
  (ZoneId/of (or (current-session/timezone)
                 DEFAULT_TIMEZONE)))

(defn now
  "Return the timezone object for the current session,
   or the default timezone object if the former can't be found."
  ^ZonedDateTime []
  (ZonedDateTime/now (get-zone)))

(defn today
  "Return the current date."
  ^LocalDate []
  (LocalDate/now))

(defn start-of-day
  "Return the current date, with the earliest possible time for this timezone."
  ^ZonedDateTime []
  (.atStartOfDay (today) (get-zone)))

(defn last-day-of-last-month
  "Create a LocalDate for today's date, and rewind it to last day of last month."
  ^LocalDate []
  (-> (today)
      (.withDayOfMonth 1)
      (.minusDays 1)))

(defn- parse-string
  "Parse a string to the correct java.time object. The string must be well-formed and
   correspond to a certain object, otherwise parsing will throw an exception."
  [^String str]
  (try
    (ZonedDateTime/parse str)
    (catch DateTimeParseException _e
      (try
        (LocalDate/parse str)
        (catch DateTimeParseException _e
          (LocalDateTime/parse str))))))

(defprotocol ConversionJavaTime
  (to-datetime [obj] [obj tz] [obj tz & options] "Convert to zoned datetime.")
  (to-localdate [obj] "Convert to date.")
  (days-between [start end] "Calculate days between dates.")
  (to-iso [obj] "Convert to iso string.")
  (to-millis [obj] "Convert date/datetime to millisecond timestamp.")
  (to-time [obj] "Convert to time."))

(extend-protocol ConversionJavaTime
  ZonedDateTime
  (to-datetime ^ZonedDateTime
    ; Create a new zoned datetime from an existing one.
    ; If a timezone is provided, convert to that timezone.
    ; If discard? flag is provided, move to new timezone without conversion.
    ([^ZonedDateTime date]
     date)
    ([^ZonedDateTime date ^ZoneId tz]
     (.withZoneSameInstant date tz))
    ([^ZonedDateTime date ^ZoneId tz & {:keys [discard?]}]
     (if discard?
       (.withZoneSameLocal date tz)
       (.withZoneSameInstant date tz))))
  (to-localdate ^LocalDate
    ; Discard time and timezone info from zoned datetime.
    [^ZonedDateTime date]
    (.toLocalDate date))
  (days-between ^Long
    ; Get number of days within the span of two zoned date times.
    [^ZonedDateTime start ^ZonedDateTime end]
    (-> (Duration/between start end) .toDays inc))
  (to-iso ^String
    ; Convert zoned datetime to iso date string.
    [^ZonedDateTime date]
    (.format date (DateTimeFormatter/ofPattern "yyyy-MM-dd")))
  (to-millis ^Long
    ; Convert zoned datetime to epoch milliseconds.
    [^ZonedDateTime datetime]
    (-> (Instant/from datetime) .toEpochMilli))
  (to-time ^LocalTime
    [date]
    (.toLocalTime date))

  LocalDate
  (to-datetime ^ZonedDateTime
    ; Create a zoned datetime from date, using session/default timezone.
    ; If a timezone is provided, use that instead.
    ; Time will be start of day according to timezone.
    ([^LocalDate date]
     (.atStartOfDay date (get-zone)))
    ([^LocalDate date ^ZoneId tz]
     (.atStartOfDay date tz))
    ([^LocalDate date ^ZoneId tz & _]
     (.atStartOfDay date tz)))
  (to-localdate ^LocalDate [^LocalDate date]
    date)
  (days-between ^Long
    ; Get number of days within the span of two dates.            
    [^LocalDate start ^LocalDate end]
    (days-between (to-datetime start) (to-datetime end)))
  (to-iso ^String
    ; Convert date to iso date string.
    [^LocalDate date]
    (.toString date))
  (to-millis ^Long
    ; Convert date to epoch milliseconds, via first converting it to zoned datetime. 
    [^LocalDate date]
    (to-millis (to-datetime date)))

  LocalDateTime
  (to-datetime ^ZonedDateTime
    ; Create a zoned datetime from datetime, using session/default timezone.
    ; If a timezone is provided, use that instead.
    ([^LocalDateTime date]
     (.atZone date (get-zone)))
    ([^LocalDateTime date ^ZoneId tz]
     (.atZone date tz))
    ([^LocalDateTime date ^ZoneId tz & _]
     (.atZone date tz)))
  (to-localdate ^LocalDate
    ; Discard time info from LocalDateTime.
    [^LocalDateTime date]
    (.toLocalDate date))
  (to-iso ^String
    ; Convert datetime to iso date string.
    [^LocalDateTime date]
    (.format date (DateTimeFormatter/ofPattern "yyyy-MM-dd")))
  (to-millis ^Long
    [^LocalDateTime date]
    (to-millis (to-datetime date)))
  (to-time ^LocalTime
    [date]
    (.toLocalTime date))

  LocalTime
  (to-time ^LocalTime
    [time]
    time)

  YearMonth
  (to-iso ^String
    ; Convert YearMonth object to iso string, with day at start of month.
    [^YearMonth ym]
    (to-iso (LocalDate/of (.getYear ym) (.getMonthValue ym) 1)))

  String
  (to-datetime ^ZonedDateTime
    ; Convert string to zoned datetime, with optional timezone.
    ; A zoned datetime string will preserve timezone, unless a new one is provided,
    ; then timezone is converted or discarded depending on discard? flag.
    ([^String str]
     (to-datetime (parse-string str)))
    ([^String str ^ZoneId tz]
     (to-datetime (parse-string str) tz))
    ([^String str ^ZoneId tz & options]
     (apply (partial to-datetime (parse-string str) tz)
            options)))
  (to-localdate ^LocalDate
    ; Convert string to LocalDate. Any time or timezone info is discarded.
    [^String str]
    (to-localdate (parse-string str)))
  (to-iso ^String
    ; Convert string to iso string by parsing it first.
    [^String str]
    (to-iso (parse-string str)))
  (to-millis ^Long
    ; Convert string to epoch milliseconds, by converting to zoned datetime first.
    [^String str]
    (to-millis (to-datetime str)))
  (to-time ^LocalTime
    ;  Convert string to local time
    [^String str]
    (LocalTime/parse str))

  Long
  (to-datetime ^ZonedDateTime
    ; Convert epoch milliseconds to zoned datetime.
    ([^Long millis]
     (-> (Instant/ofEpochMilli millis)
         (.atZone (get-zone))))
    ([^Long millis ^ZoneId tz]
     (-> (Instant/ofEpochMilli millis)
         (.atZone tz)))
    ([^Long millis ^ZoneId tz & _]
     (-> (Instant/ofEpochMilli millis)
         (.atZone tz))))
  (to-localdate ^LocalDate
    ; Convert epoch milliseconds to date, by converting to zoned datetime first.
    [^Long millis]
    (to-localdate (to-datetime millis)))

  org.joda.time.LocalDate
  (to-datetime ^ZonedDateTime
    ([date]
     (to-datetime (.toString date)))
    ([date ^ZoneId tz]
     (to-datetime (.toString date) tz))
    ([date ^ZoneId tz & options]
     (apply (partial to-datetime (.toString date) tz)
            options)))
  (to-localdate ^LocalDate [date]
    (to-localdate (.toString date)))
  (to-iso ^String [date]
    (.toString date))

  org.joda.time.LocalDateTime
  (to-datetime ^ZonedDateTime
    ([date]
     (to-datetime (.toString date)))
    ([date ^ZoneId tz]
     (to-datetime (.toString date) tz))
    ([date ^ZoneId tz & options]
     (apply (partial to-datetime (.toString date) tz)
            options)))
  (to-localdate ^LocalDate [date]
    (to-localdate (.toString date)))
  (to-iso ^String [date]
    (.toString date "yyyy-MM-dd"))
  (to-millis ^Long [date]
    (.getMillis date))
  (to-time ^LocalTime
    [date]
    (-> date .toLocalTime .toString LocalTime/parse))

  org.joda.time.DateTime
  (to-datetime ^ZonedDateTime
    ([date]
     (to-datetime (.toString date)))
    ([date ^ZoneId tz]
     (to-datetime (.toString date) tz))
    ([date ^ZoneId tz & options]
     (apply (partial to-datetime (.toString date) tz)
            options)))
  (to-localdate ^LocalDate [date]
    (to-localdate (.toString date)))
  (to-iso ^String [date]
    (.toString date "yyyy-MM-dd"))
  (to-millis ^Long [date]
    (.getMillis date))
  (to-time ^LocalTime
    [date]
    (-> date .toLocalTime .toString LocalTime/parse))

  org.joda.time.LocalTime
  (to-time ^LocalTime
    [time]
    (-> time .toString LocalTime/parse)))

(defn format
  "Format java.time object according to custom format string."
  ^String [d f]
  (when d
    (.format d (DateTimeFormatter/ofPattern f))))

(def rfc-1123-formatter
  (-> (DateTimeFormatter/ofPattern "EEE, dd MMM yyyy HH:mm:ss 'GMT'")
      (.withLocale java.util.Locale/US)
      (.withZone ZoneOffset/UTC)))

(defn to-rfc-1123 ^String [d]
  (when d
    (.format d rfc-1123-formatter)))

(defn duration->hours [d]
  (let [d (cond
            (instance? org.joda.time.ReadableDuration d) (.getMillis d)
            (instance? Duration d) (.toMillis d)
            :else (-> (long d) org.joda.time.Duration. .getMillis))]
    (with-precision 16
      (/ d 3600000M))))

(defn first-time-after [date-time time]
  (let [new-date-time (.with date-time time)]
    (if (.isBefore new-date-time date-time)
      (recur date-time (.plusDays new-date-time 1))
      new-date-time)))

(defn date-seq
  ([start-date end-date] (date-seq start-date end-date 1))
  ([start-date end-date step]
   (when (.isBefore start-date end-date)
     (cons
      start-date
      (lazy-seq
       (date-seq (.plusDays start-date step) end-date step))))))
