(ns com.timezynk.useful.channel
  (:require
   [clojure.core.reducers :as r]
   [clojure.tools.logging :as log :refer [debug info warn]]
   [clojure.string :as string]
   [com.timezynk.useful.map :as umap]
   [com.timezynk.useful.mongo :as um]
   [com.timezynk.useful.prometheus.core :as prometheus]
   [somnium.congomongo :as mongo])
  (:import [java.util.concurrent LinkedBlockingQueue
            PriorityBlockingQueue
            BlockingQueue
            TimeUnit]
           [java.util UUID]))

(defonce ^{:dynamic true} *debug* true)

(def ^{:dynamic true} *reply-channel* nil)

(def ^:const NUM_WORKERS 2)

(defonce subscribers (ref {}))

(defonce current-task-id (atom 0))

(defonce current-message-id (atom 0))

(def queue-size (prometheus/gauge :channel_queue_size "Number of actions waiting in the channel queue" :queue_id))
(def processed-messages (prometheus/counter :channel_processed_total "Number of actions processed by the channel queue" :queue_id))

(defrecord ChannelMessage [prio task]
  Comparable
  (compareTo [this o]
    (compare (:prio this) (:prio o))))

(defn- signature [context topic cname & {:keys [message]}]
  (let [new-id (get-in message [0 :id])]
    (format "[%s] %s in %s"
            (or context "N/A")
            (cond-> (name topic)
              new-id (str " " new-id))
            (name cname))))

(defn- log-event [task event]
  (let [{:keys [context topic cname message]} task]
    (format "%s <%s>"
            (signature context topic cname :message message)
            (string/upper-case (name event)))))

(defprotocol MessageTask
  (process [task channel] "Process the message task"))

(defn- run-subscriber [task subscriber]
  (info (log-event task :start))
  (let [{:keys [context topic cname message]} task
        start-time (System/currentTimeMillis)
        result ((:f subscriber) topic cname context message)
        end-time (System/currentTimeMillis)]
    (info (format "%s (%d msecs)"
                  (log-event task :finish)
                  (- end-time start-time)))
    result))

(defrecord RequestResponseTask [subscriber task-id topic cname message reply-channel context]
  MessageTask
  (process [task channel]
    (.put reply-channel [:started task-id])
    (debug topic cname "running request-response task" task-id)
    (try
      ; Bind dynamic reply channel so that recursive tasks are collected in the outermost
      ; wait-for
      (binding [*reply-channel* reply-channel]
        (let [result (run-subscriber task subscriber)]
          (.put reply-channel [:finished task-id result])))
      (catch Exception e
        (warn e topic cname "request-response failed to run")
        (.put reply-channel [:exception task-id e])))))

(defrecord BroadcastTask [subscriber topic cname message context]
  MessageTask
  (process [task channel]
    (debug topic cname "running broadcast task")
    (try
      (run-subscriber task subscriber)
      (catch Exception e
        (warn e topic cname "broadcast failed to run")))))

(defprotocol Subscriber
  (publish [this topic cname message reply-channel context] "Publish message to subscriber"))

(defn- eligible?
  "True if subscriber is eligible to receive events from collection cname,
   false otherwise."
  [subscriber cname]
  (let [collection-name (:collection-name subscriber)]
    (or (nil? collection-name)
        (= collection-name cname))))

(defrecord RequestResponseSubscriber [collection-name f]
  Subscriber
  (publish [this topic cname message reply-channel context]
    (let [task-id (swap! current-task-id inc)
          task (RequestResponseTask. this task-id topic cname message reply-channel context)]
      (info (log-event task :enqueue))
      (.put reply-channel [:queued task-id])
      (ChannelMessage. 5 task))))

(defrecord BroadcastSubscriber [collection-name f]
  Subscriber
  (publish [this topic cname message _reply-channel context]
    (let [task (BroadcastTask. this topic cname message context)]
      (info (log-event task :enqueue))
      (ChannelMessage. 10 task))))

(defn ^BlockingQueue publish! [^BlockingQueue channel context topic cname messages]
  (when (seq messages)
    (let [reply-channel (LinkedBlockingQueue.)
          eligible-subscribers (->> topic
                                    (get @subscribers)
                                    (filter #(eligible? % cname)))]
      (info (format "%s (%d message(s), %d subscriber(s))"
                    (signature context topic cname)
                    (count messages)
                    (count eligible-subscribers)))
      (doseq [message messages
              s eligible-subscribers]
        (try
          (when-let [msg (publish s
                                  topic
                                  cname
                                  message
                                  (or *reply-channel* reply-channel)
                                  context)]
            (.put channel msg))
          (catch Exception e
            (warn e topic cname "failed to publish" message))))
      (.put reply-channel [:queued-message-tasks])
      reply-channel)))

(defn wait-for [timeout-ms reply-channel]
  (when reply-channel
    (loop [[event id payload] (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
           tasks #{}]
      (debug "event" event id "waiting for" tasks)
      (case event
        :queued-message-tasks (if (empty? tasks)
                                (do
                                  (debug "completed. No tasks to wait for")
                                  true)
                                (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                       tasks))

        :queued               (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                     (conj tasks id))

        :started              (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                     tasks)

        :finished             (let [new-tasks (disj tasks id)]
                                (if (empty? new-tasks)
                                  (do
                                    (debug "completed. All tasks finished")
                                    true)
                                  (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                                         new-tasks)))

        :exception            (throw payload)

        (if (seq tasks)
          (do
            (info "timeout. Still waiting for" tasks)
            false)
          (recur (.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)
                 tasks))))))

(defn- subscribe [topic subscriber]
  (dosync
   (alter subscribers update-in [topic] conj subscriber)))

(defn subscribe-broadcast [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new broadcast subscriber")
    (if (sequential? topic)
      (doseq [t topic]
        (subscribe t (BroadcastSubscriber. collection-name f)))
      (subscribe topic (BroadcastSubscriber. collection-name f)))))

(defn subscribe-request-response [topic collection-name f]
  (when (and topic f)
    (debug topic collection-name "new request-response subscriber")
    (if (sequential? topic)
      (doseq [t topic]
        (subscribe t (RequestResponseSubscriber. collection-name f)))
      (subscribe topic (RequestResponseSubscriber. collection-name f)))))

(defn unsubscribe-all []
  (dosync
   (ref-set subscribers {})))

(defn route-message [message channel message-counter]
  (when-let [t (:task message)]
    (process t channel))
  (.inc message-counter))

(defn broker-loop [^BlockingQueue channel queue-id]
  (fn []
    (info "starting message broker")
    (let [size-gauge (prometheus/gauge-with-labels queue-size queue-id)
          message-counter (prometheus/counter-with-labels processed-messages queue-id)]
      (while true
        (try
          (route-message (.take channel) channel message-counter)
          (.set size-gauge (.size channel))
          (catch Exception e
            (warn e "Exception in channel broker")
            (Thread/sleep 100)))))))

(defn ^BlockingQueue create-broker! [^BlockingQueue channel queue-id]
  (dotimes [i NUM_WORKERS]
    (doto (Thread. (broker-loop channel queue-id) (str "mchan-" i))
      (.setDaemon true)
      (.start)))
  channel)

(defn ^BlockingQueue start-channel! []
  (create-broker! (PriorityBlockingQueue.) (str (UUID/randomUUID))))

