(ns com.timezynk.useful.date
  (:import [org.joda.time LocalDateTime LocalDate LocalTime Interval Duration
            DateTime DateTimeZone ReadablePartial ReadableDateTime
            ReadableDuration IllegalInstantException DateTimeConstants]
           [org.joda.time.format DateTimeFormat DateTimeFormatter]))

(def ^DateTimeFormatter iso-formatter
  (DateTimeFormat/forPattern "yyyy-MM-dd"))

(def ^DateTimeFormatter rfc-1123-formatter
  (.. (DateTimeFormat/forPattern "EEE, dd MMM yyyy HH:mm:ss 'GMT'")
      (withLocale java.util.Locale/US)
      (withZone DateTimeZone/UTC)))

(defprotocol ToISODateConvertible
  (to-iso [d] "Convert to iso date (yyyy-MM-dd)"))

(def iso-pattern #"(\d{4}-\d{2}-\d{2})")
(def date-time-pattern #"(\d{4}-\d{2}-\d{2}T\d{2}\:\d{2}\:\d{2}\.\d{3})")

(defn date-string? [v]
  (and (string? v) (re-matches iso-pattern v)))

(defn date-time-string? [v]
  (and (string? v) (re-matches date-time-pattern v)))

(extend-protocol ToISODateConvertible
  LocalDateTime
  (to-iso [d] (.print iso-formatter d))

  LocalDate
  (to-iso [d] (.print iso-formatter d))

  java.time.LocalDateTime
  (to-iso [d] (.toString (.toLocalDate d)))

  java.time.LocalDate
  (to-iso [d] (.toString d))

  ReadableDateTime
  (to-iso [d] (.print iso-formatter d))

  String
  (to-iso [d]
    (when-let [m (re-find iso-pattern d)]
      (get m 1)))

  nil
  (to-iso [d] nil))

(defn to-rfc-1123
  "Convert to RFC 1123 formatted date"
  [d]
  (when d
    (.print rfc-1123-formatter d)))

(defn parse-rfc-1123
  "Parse date in RFC 1123 format"
  [s]
  (when s
    (.parseDateTime rfc-1123-formatter s)))

(defprotocol LocalDateTimeConvertible
  (->local-datetime [d] "Convert to Joda LocalDateTime"))

(extend-protocol LocalDateTimeConvertible
  String
  (->local-datetime [s] (LocalDateTime/parse s))

  LocalDateTime
  (->local-datetime [d] d)

  java.time.LocalDateTime
  (->local-datetime [d] (LocalDateTime/parse (.toString d)))

  ReadableDateTime
  (->local-datetime [d] (.toLocalDateTime (.toDateTime d)))

  nil
  (->local-datetime [d] d))

(defprotocol LocalDateConvertible
  (->local-date [d] "Convert to Joda LocalDate"))

(extend-protocol LocalDateConvertible
  String
  (->local-date [s] (LocalDate/parse s))

  LocalDateTime
  (->local-date [d] (.toLocalDate d))

  LocalDate
  (->local-date [d] d)

  java.time.LocalDateTime
  (->local-date [d] (LocalDate/parse (.toString (.toLocalDate d))))

  java.time.LocalDate
  (->local-date [d] (LocalDate/parse (.toString d)))

  ReadableDateTime
  (->local-datetime [d] (.toLocalDate (.toDateTime d)))

  nil
  (->local-date [d] d))

(defprotocol LocalTimeConvertible
  (->local-time [d] "Convert to Joda LocalTime"))

(extend-protocol LocalTimeConvertible
  String
  (->local-time [s] (LocalTime/parse s))

  LocalDateTime
  (->local-time [d] (.toLocalTime d))

  LocalTime
  (->local-time [d] d)

  java.time.LocalDateTime
  (->local-time [d] (LocalTime/parse (.toString (.toLocalTime d))))

  java.time.LocalTime
  (->local-time [d] (LocalTime/parse (.toString d)))

  nil
  (->local-time [d] d))

(defn ^LocalDateTime read-iso [s]
  (when s
    (.parseLocalDateTime iso-formatter s)))

(defn first-time-after [^LocalDateTime date-time ^ReadablePartial time]
  (let [new-date-time (.withFields date-time time)]
    (if (.isBefore new-date-time date-time)
      (recur date-time (.plusDays new-date-time 1))
      new-date-time)))

(defn date-seq
  ([start-date end-date] (date-seq start-date end-date 1))
  ([^LocalDateTime start-date ^ReadablePartial end-date step]
   (when (.isBefore start-date end-date)
     (cons
      start-date
      (lazy-seq
       (date-seq (.plusDays start-date step) end-date step))))))

(defn ^DateTime parse-datetime [datetime-str ^DateTimeZone timezone]
  (let [ldt (LocalDateTime/parse datetime-str)]
    (try
      (.toDateTime ldt timezone)
      (catch IllegalInstantException e
        (parse-datetime (str (.plusHours ldt 1)) timezone)))))

(defn ^Interval to-interval [{:keys [start end]} timezone]
  (when (and start end)
    (let [start (parse-datetime start timezone)
          end (parse-datetime end timezone)]
      (Interval. start end))))

(defn ^Duration to-duration [^Interval interval]
  (when interval
    (.toDuration interval)))

(defn ^Duration add-duration [^Duration a ^Duration b]
  (if a
    (if b
      (.plus a b)
      a)
    b))

(defn duration->hours [d]
  (let [^ReadableDuration d (if (instance? ReadableDuration d) d (Duration. (long d)))]
    (with-precision 16
      (/ (.getMillis d) 3600000M))))

(defn next-weekday
  "Move date to next weekday (Mon-Fri)"
  [^LocalDate d]
  (condp = (.getDayOfWeek d)
    DateTimeConstants/SATURDAY (.plusDays d 2)
    DateTimeConstants/SUNDAY (.plusDays d 1)
    d))
