(ns com.timezynk.useful.nsq
  (:require [clojure.tools.logging :as log]
            [cheshire.core :as json])
  (:import [com.sproutsocial.nsq Publisher Subscriber MessageHandler]))

(defn  create-publish-api
  "Creates a com.sproutsocial.nsq.Publisher object."
  ^Publisher [servers config]
  (log/info "Connecting NSQ publisher to" servers)
  (let [^Publisher publisher (Publisher. servers)]
    (.setConfig publisher config)
    publisher))

(defn publish
  "Asynchronously serializes and publishes message on topic through api.
   Returns the operation wrapped in a future."
  [api topic message]
  (future
    (try (let [message (-> message json/encode (.getBytes "UTF-8"))]
           (.publishBuffered api topic message))
         (catch Exception e
           (log/warn e "NSQ publish failed for" (:op message)
                     "in" (:coll message))))))

(defonce ^:private subscribers
  (atom nil))

(defn create-subscribe-api
  "Creates a com.sproutsocial.nsq.Subscriber object."
  ^Subscriber [lookup-server config]
  (let [api (Subscriber. (into-array String [lookup-server]))]
    (.setConfig api config)
    api))

(defn create-message-handler
  "Turns function f into a Java object which implements MessageHandler."
  [f]
  (reify MessageHandler
    (accept [_this message]
      (try (f (.getTopic message) (slurp (.getData message)))
           (catch Throwable t
             (log/error t "NSQ handler failed"))
           (finally (.finish message))))))

(defn- subscribe*
  "Implementation detail."
  [api topic channel f]
  (try (.subscribe api topic channel (create-message-handler f))
       (catch Throwable t
         (log/warn t "NSQ subscribe failed"))))

(defn subscribe
  "Idempotently subscribes f to topic on channel through api.
   f must take two string arguments: topic and message."
  [api topic channel f]
  (swap! subscribers
         (fn [registry]
           (update-in registry
                      [topic channel f]
                      #(or % (future (subscribe* api topic channel f))))))
  (-> @subscribers (get-in [topic channel f]) deref))
