(ns com.timezynk.useful.core
  (:require [clojure.string :as s]))

(defn timestamp []
  (-> (System/currentTimeMillis) (/ 1000) (int)))

(defn concat-bytes [^"[B" a ^"[B" b]
  (let [alen (alength a)
        blen (alength b)
        res (byte-array (+ alen blen))]
    (System/arraycopy a 0 res 0 alen)
    (System/arraycopy b 0 res alen blen)
    res))

(defn split-bytes
  "Split byte array at index pos"
  [^"[B" bytes pos]
  (let [len (alength bytes)]
    (if (<= len pos)
      bytes
      (let [a (byte-array pos)
            b (byte-array (- len pos))]
        (System/arraycopy bytes 0 a 0 (alength a))
        (System/arraycopy bytes pos b 0 (alength b))
        [a b]))))

(def random-string
  (let [ascii-codes (concat (range 48 58) (range 66 91) (range 97 123))]
    (fn [len]
      (apply str
        (repeatedly len #(char (rand-nth ascii-codes)))))))

(def SERVER_FORMAT #"^\s*([^:]+)(?::(\d+))?\s*$")

(defn server-description [acc server]
  (if-let [m (re-matches SERVER_FORMAT server)]
    (conj acc
      (merge
        {:host (get m 1)}
        (when-let [port (get m 2)]
          {:port (Long/parseLong port)})))
    acc))

(defn parse-server-string [servers]
  (when (and servers (string? servers))
    (reduce server-description [] (s/split servers #"[ ,]"))))

(defn parse-env
  ([env-var] (parse-env env-var "127.0.0.1"))
  ([env-var fallback-value]
    (parse-server-string (or (System/getenv env-var) fallback-value))))
