(ns com.timezynk.useful.rest
  (:use [slingshot.slingshot :only [throw+]])
  (:require
    [cheshire.core :as json]
    [clojure.core.reducers :as r]
    [clojure.java.io :as io]
    [clojure.spec.alpha :as spec]
    [com.timezynk.useful.base64 :as b64]
    [com.timezynk.useful.date :as ud]
    [com.timezynk.useful.hash :as uh]
    [com.timezynk.useful.mongo :as um]
  )
  (:import [org.joda.time DateTime]))

(defn error [status message]
  (throw+ {:code status :error message}))

(defn unauthorized []
  (error 401 "Unauthorized"))

(defn not-found []
  (error 404 "Not found"))

(defn json-response [body & {:keys [status headers] :or {status 200}}]
  (if body
    {:status status
     :headers
      (merge
        {"Content-Type" "application/json"
         "Expires" "-1"
         "Cache-Control" "no-cache"}
        headers)
     :body body}
    (not-found)))

(defn file-response [body & {:keys [filename content-type content-description status]
                             :or {status 200}}]
  (if body
    {:status 200
     :headers
     {"Content-Disposition" (str "attachment; filename=" filename)
      "Content-Type" content-type
      "Content-Description" content-description}
     :body body}
    (not-found)))

(defn etag-response [req response]
  (if response
    (let [if-none-match (get-in req [:headers "if-none-match"])
          response      (if (string? response) response (json/encode response))
          body          (.getBytes response "UTF-8")
          etag          (str "\"" (b64/base64-str (uh/sha256 body)) "\"")
          headers       {"ETag" etag
                         "Cache-Control" "max-age=0,private"
                         "Expires" (ud/to-rfc-1123 (DateTime.))
                         "Content-Type" "application/json; charset=utf-8"}]
      (if (and if-none-match etag (= if-none-match etag))
        {:status 304
         :body ""
         :headers headers}
        {:status 200
         :body (io/input-stream body)
         :headers (assoc headers "Content-Length" (count body))}))
    (not-found)))

(defn integer-param [params k default]
  (if-let [v (get params k)]
    (Long/parseLong v)
    default))

(defn remove-nil [u]
  (r/reduce
    (fn [acc k v]
      (if v
        (assoc acc k v)
        acc))
    {} u))
