(ns com.timezynk.useful.cancan
  (:require
   [slingshot.slingshot :refer [try+ throw+]]))

(def ^{:dynamic true} *ability*)

(def ^:dynamic *restrictors* nil)

(defn can
  ([action object]
   (can action object (constantly true) *ability*))
  ([action object pred]
   (can action object pred *ability*))
  ([action object pred ability]
   (let [actions (if (sequential? action) action [action])
         predicates (zipmap actions (repeat pred))]
     (swap! ability update-in [object] merge predicates))))

(defn eligible
  "Declares that `action` may be performed on entities gotten by restricting
   the `object` query with `restrictor`."
  [action object restrictor]
  (let [actions (cond-> action
                  (not (sequential? action)) (vector))
        restrictors (zipmap actions (repeat restrictor))]
    (swap! *restrictors* update object merge restrictors)))

(defn in-scope?
  "True if the calling code and the calling thread have been primed for
   authorization checks, false otherwise.

   Priming for authorization involves a series of cancan/can calls within a
   cancan/with-ability scope."
  []
  (bound? #'*ability*))

(defn can?
  ([action object]
   (can? action object nil *ability*))
  ([action object instance]
   (can? action object instance *ability*))
  ([action object instance ability]
   (let [pred (or (get-in @ability [:all :all])
                  (get-in @ability [:all action])
                  (get-in @ability [object action])
                  (get-in @ability [object :all]))]
     (when pred
       (pred action object instance)))))

(defn downscope
  "If `action` on `object` has been registered, restricts `instance` to the
   subset of eligible entities. Otherwise, returns `instance`."
  [action object instance]
  (let [restriction (or (get-in @*restrictors* [:all :all])
                        (get-in @*restrictors* [:all action])
                        (get-in @*restrictors* [object action])
                        (get-in @*restrictors* [object :all]))]
    (cond->> instance
      restriction (restriction action object))))

(defn authorize!
  ([action object] (authorize! action object nil *ability*))
  ([action object instance] (authorize! action object instance *ability*))
  ([action object instance ability]
   (if (can? action object instance ability)
     instance
     (throw+ {:error :cancan :action action :object object :instance instance}))))

(defn authorize-all!
  ([action object instances] (authorize-all! action object instances *ability*))
  ([action object instances ability]
   (when-not (every? true? (map #(can? action object % ability) instances))
     (throw+ {:error :cancan :action action :object object :instance instances}))
   instances))

(defmacro with-ability [& body]
  `(binding [*ability* (atom {})
             *restrictors* (atom {})]
     ~@body))

(defn wrap-cancan [handler]
  (fn [request]
    (try+
     (with-ability
       (handler request))
     (catch [:error :cancan] _e
       (throw+ {:code 401 :error "Unauthorized"})))))
