(ns com.timezynk.useful.prometheus.core
  (:import [io.prometheus.client
            CollectorRegistry
            Counter
            Counter$Child
            Gauge
            Gauge$Child
            Histogram
            Histogram$Child]
           [io.prometheus.client.hotspot DefaultExports]
           [java.util UUID]))

(defonce backend-id (str (UUID/randomUUID)))

(def PROMETHEUS_NAMESPACE (or (System/getenv "PROMETHEUS_NAMESPACE") "tzbackend"))

(defn counter-with-labels ^Counter$Child [^Counter counter & labels]
  (.labels counter (into-array String (conj labels backend-id))))

(defn histogram-with-labels ^Histogram$Child [^Histogram histogram & labels]
  (.labels histogram (into-array String (conj labels backend-id))))

(defn gauge-with-labels ^Gauge$Child [^Gauge histogram & labels]
  (.labels histogram (into-array String (conj labels backend-id))))

(defn inc! [counter & labels]
  (when-let [child (apply counter-with-labels counter labels)]
    (.inc child)))

(defn inc-by! [counter value & labels]
  (when-let [child (apply counter-with-labels counter labels)]
    (.inc child (double value))))

(defn observe! [^Histogram histogram value & labels]
  (when-let [child (apply histogram-with-labels histogram labels)]
    (.observe child (double value))))

(defn counter [n h & label-names]
  (.. (Counter/build)
      (namespace PROMETHEUS_NAMESPACE)
      (name (name n))
      (help h)
      (labelNames (into-array String (conj (map name label-names) "backend_id")))
      (register)))

(defn gauge [n h & label-names]
  (.. (Gauge/build)
      (namespace PROMETHEUS_NAMESPACE)
      (name (name n))
      (help h)
      (labelNames (into-array String (conj (map name label-names) "backend_id")))
      (register)))

(defn histogram [n h buckets & label-names]
  (.. (Histogram/build)
      (buckets (double-array buckets))
      (namespace PROMETHEUS_NAMESPACE)
      (name (name n))
      (help h)
      (labelNames (into-array String (conj (map name label-names) "backend_id")))
      (register)))

(defn init-default-exports []
  (DefaultExports/initialize))

(defn unregister [collector]
  (-> (CollectorRegistry/defaultRegistry)
      (.unregister collector)))
