(ns com.timezynk.useful.channel.subscriber
  (:require [com.timezynk.useful.channel.message :as message]
            [com.timezynk.useful.channel.task :refer [->RequestResponseTask
                                                      ->BroadcastTask]]))

(defonce ^:private subscribers (ref {}))

(defonce ^:private current-task-id (atom 0))

(defprotocol Subscriber
  (publish [this topic cname message reply-channel context]
           "Publish message to subscriber"))

(defn- eligible?
  "True if subscriber is eligible to receive events from collection cname,
   false otherwise."
  [subscriber cname]
  (let [collection-name (:collection-name subscriber)]
    (or (nil? collection-name)
        (= collection-name cname))))

(defn eligible-for
  "Sequence of subscribers eligible for topic on cname."
  [topic cname]
  (->> topic
       (get @subscribers)
       (filter #(eligible? % cname))))

(defrecord RequestResponseSubscriber [collection-name f]
  Subscriber
  (publish [this topic cname message reply-channel context]
    (let [task-id (swap! current-task-id inc)]
      (.put reply-channel [:queued task-id])
      (message/make 5 (->RequestResponseTask this
                                             task-id
                                             topic
                                             cname
                                             message
                                             reply-channel
                                             context)))))

(defrecord BroadcastSubscriber [collection-name f]
  Subscriber
  (publish [this topic cname message _reply-channel context]
    (message/make 10
                  (->BroadcastTask this topic cname message context))))

(defn add [topic subscriber]
  (dosync
    (alter subscribers update-in [topic] conj subscriber)))

(defn remove-all []
  (dosync
    (ref-set subscribers {})))
