(ns com.timezynk.useful.mongo
  (:require
   [clojure.core.reducers :as r]
   [clojure.set :as cs]
   [com.timezynk.useful.date :as date]
   [potemkin :as pk]
   [slingshot.slingshot :refer [throw+]]
   [somnium.congomongo :as mongo]
   [somnium.congomongo.coerce :as mongo-coerce]
   [validateur.validation :as validation])
  (:import [org.bson BsonType]
           [org.bson.codecs Codec]))

(def ^:const OBJECT_ID_REGEX #"^[\da-f]{24}$")

(defprotocol PotentialObjectId
  (object-id [o] "Convert to object id"))

(extend-protocol PotentialObjectId
  org.bson.types.ObjectId
  (object-id [o] o)
  clojure.lang.Named
  (object-id [o] (org.bson.types.ObjectId. (name o)))
  Object
  (object-id [o] (org.bson.types.ObjectId. (str o)))
  nil
  (object-id [_o] nil))

(extend-protocol mongo-coerce/ConvertibleToMongo
  java.time.LocalDate
  (clojure->mongo
    ; Make congomongo convert java.time.LocalDate to string.
    ^String [^java.time.LocalDate d]
    (date/to-iso d))

  java.time.LocalTime
  (clojure->mongo
    ; Make congomongo convert java.time.LocalTime to string.
    ^String [^java.time.LocalTime t]
    (.toString t))

  java.time.LocalDateTime
  (clojure->mongo
    ; Make congomongo convert java.time.LocalDateTime to storable date.
    ^java.util.Date [^java.time.LocalDateTime dt]
    (date/to-utildate dt))

  java.time.ZonedDateTime
  (clojure->mongo
    ; Make congomongo convert java.time.ZonedDateTime to storable date.
    ^java.util.Date [^java.time.ZonedDateTime zdt]
    (date/to-utildate zdt))

  org.joda.time.LocalDateTime
  (clojure->mongo [o] (str o))

  org.joda.time.LocalTime
  (clojure->mongo [o] (str o))

  org.joda.time.LocalDate
  (clojure->mongo [o] (str o)))

(extend-protocol mongo-coerce/ConvertibleFromMongo
  java.util.Date
  (mongo->clojure [o _keywordize]
    (date/to-datetime o)))

;; New MongoDB API can handle date-times
(defn joda-datetime-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      org.joda.time.DateTime)))

(defn joda-localdate-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      org.joda.time.LocalDate)))

(defn joda-localdatetime-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      org.joda.time.LocalDateTime)))

(defn joda-localtime-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      org.joda.time.LocalTime)))

(defn java-time-localdate-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      java.time.LocalDate)))

(defn java-time-localtime-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeString writer (.toString value)))

    (getEncoderClass [_this]
      java.time.LocalTime)))

(defn java-time-localdatetime-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeDatetime writer (date/to-millis value)))

    (getEncoderClass [_this]
      java.time.LocalDateTime)))

(defn java-time-zoneddatetime-codec []
  (reify Codec
    (decode [_this reader _decoder-context]
      (date/to-datetime (.readDatetime reader)))

    (encode [_this writer value _encoder-context]
      (.writeDatetime writer (date/to-millis value)))

    (getEncoderClass [_this]
      java.time.ZonedDateTime)))

(defn java-util-date-codec []
  (reify Codec
    (decode [_this _reader _decoder-context]
      ())

    (encode [_this writer value _encoder-context]
      (.writeDatetime writer (.getTime value)))

    (getEncoderClass [_this]
      java.util.Date)))

(defonce new-codecs [(joda-datetime-codec)
                     (joda-localdate-codec)
                     (joda-localdatetime-codec)
                     (joda-localtime-codec)
                     (java-time-localdate-codec)
                     (java-time-localtime-codec)
                     (java-time-localdatetime-codec)
                     (java-time-zoneddatetime-codec)
                     (java-util-date-codec)])

(defonce new-types {BsonType/DATE_TIME java.time.ZonedDateTime})
;; -------

(defn object-ids [ids]
  (when ids
    (map object-id ids)))

(defn object-id? [s]
  (or
   (instance? org.bson.types.ObjectId s)
   (and (string? s) (re-matches OBJECT_ID_REGEX s))))

(defn unpack-id [entry]
  (cs/rename-keys entry {:_id :id}))

(defn pack-id [entry]
  (cs/rename-keys entry {:id :_id}))

(defn unpack-ids [entries]
  (map unpack-id entries))

(defn copy-attr
  ([e k] (copy-attr e k object-id))
  ([e k f]
   (when (contains? e k)
     {k (f (get e k))})))

(defmacro map-copy-attr
  ([e ks]
   `(apply merge (map (fn [k#] (copy-attr ~e k#)) ~ks)))
  ([e ks f]
   `(apply merge (map (fn [k#] (copy-attr ~e k# ~f)) ~ks))))

(defn object-ids-q [k ids]
  (when ids
    {k {:$in (object-ids ids)}}))

(defn object-id-q [k id]
  (when id
    {k (object-id id)}))

(defprotocol DoubleConvertible
  (to-double [d] "Convert to double"))

(extend-protocol DoubleConvertible
  String
  (to-double [s]
    (try
      (Double/parseDouble s)
      (catch NumberFormatException _e
        nil)))
  Double
  (to-double [d] d)
  Integer
  (to-double [i] (.doubleValue i))
  nil
  (to-double [_o] nil))

(defprotocol ConvertibleToLong
  (to-long [o] "Convert to long"))

(extend-protocol ConvertibleToLong
  String
  (to-long [s] (Long/parseLong s))
  Long
  (to-long [l] l)
  Integer
  (to-long [i] (.longValue i)))

(defn nil-empty-strings [a]
  (r/reduce
   (fn [a k v]
     (if (and (string? v) (.isEmpty ^String v))
       (assoc a k nil)
       a))
   a a))

(defn get-id-in [m ks]
  (object-id (get-in m ks)))

(defn get-ids-in [m ks]
  (object-ids (get-in m ks)))

(defn extend-defaults [defaults entry]
  (merge defaults (select-keys entry (keys defaults))))

(defn validate-string [& strings]
  (every? #(and % (string? %) (seq %)) strings))

(defn validate-object-id [& ids]
  (every?
   #(try
      (object-id %)
      (catch Exception _e
        nil)) ids))

(defmacro defvalidator [fn-name & body]
  `(def ~fn-name
     (let [vs# (validation/validation-set ~@body)]
       (fn [entry#]
         (let [errors# (vs# entry#)]
           (if (empty? errors#)
             entry#
             (throw+ {:code 400 :error :validation :message errors#})))))))

(defn validate-all [vs entries]
  (r/reduce
   (fn [result e]
     (merge-with cs/union result (vs e)))
   {} entries))

(pk/import-fn validation/exclusion-of)
(pk/import-fn validation/format-of)
(pk/import-fn validation/inclusion-of)
(pk/import-fn validation/length-of)
(pk/import-fn validation/numericality-of)
(pk/import-fn validation/presence-of)
(pk/import-fn validation/acceptance-of)

(defn object-id-of [attribute]
  (fn [e]
    (let [v (get e attribute)]
      (if (and v (validate-object-id v))
        [true {}]
        [false {attribute #{(str "Invalid ObjectId in " attribute)}}]))))

(defmacro child-of [attribute & body]
  `(let [a# ~attribute
         vs# (validation/validation-set ~@body)]
     (fn [e#]
       (let [errors# (vs# (get e# a#))]
         (if (empty? errors#)
           [true {}]
           [false {a# #{errors#}}])))))

(defmacro children-of [attribute & body]
  `(let [a# ~attribute
         vs# (validation/validation-set ~@body)]
     (fn [e#]
       (let [errors# (validate-all vs# (get e# a#))]
         (if (empty? errors#)
           [true {}]
           [false {a# #{errors#}}])))))

(defn fetch [& args]
  (unpack-ids (apply mongo/fetch args)))

(defn insert! [collection entry]
  (unpack-id (mongo/insert! collection entry)))

(defn insert-and-wait! [collection entry]
  (unpack-id
   (mongo/insert! collection entry :write-concern :acknowledged)))

(defn find-by-id [collection id]
  (try
    (unpack-id (mongo/fetch-by-id collection (object-id id)))
    (catch IllegalArgumentException _e
      nil)))

(defn intersecting-query
  ([from to] (intersecting-query from to :start :end))
  ([from to start-key end-key]
   {start-key {:$lt (date/to-datetime to)}
    end-key   {:$gte (date/to-datetime from)}}))

(defn start-inside-period-query
  ([from to] (start-inside-period-query from to :start))
  ([from to start-key]
   {start-key
    (merge
     (when from {:$gte (date/to-datetime from)})
     (when to {:$lt (date/to-datetime to)}))}))
