(ns com.timezynk.useful.channel
  (:require
   [clojure.tools.logging :as log]
   [com.timezynk.useful.channel.group :as group]
   [com.timezynk.useful.channel.logging :as chlog]
   [com.timezynk.useful.channel.subscriber :as subscriber]
   [com.timezynk.useful.channel.task :as task]
   [com.timezynk.useful.env :as env])
  (:import [java.util.concurrent LinkedBlockingQueue
            PriorityBlockingQueue
            BlockingQueue
            TimeUnit]
           [java.util UUID]))

(def ^:private ^:const NUM_REQUEST_RESPONSE_WORKERS 2)

(def ^:private ^:const NUM_BROADCAST_WORKERS 2)

(def ^:private ^:const BROADCAST_THROTTLE_INTERVAL_MS 50)

(defn- publish* ^BlockingQueue
  [channel context topic collection messages]
  (when (seq messages)
    (let [reply-channel (LinkedBlockingQueue.)]
      (doseq [m messages
              s (subscriber/eligible-for channel topic collection)]
        (try
          (.put (:queue channel)
                (subscriber/publish s
                                    topic
                                    collection
                                    m
                                    (or task/*reply-channel* reply-channel)
                                    context))
          (catch Exception e
            (log/warn e topic collection "failed to publish" m))))
      (.put reply-channel [:queued-message-tasks])
      reply-channel)))

(defn wait-for
  "Awaits responses on `reply-channel`.
   Returns `false` on timeout.
   Throws the caught exception on error.
   Returns `true` otherwise."
  [timeout-ms reply-channel]
  (when reply-channel
    (let [poll #(.poll reply-channel timeout-ms TimeUnit/MILLISECONDS)]
      (loop [[event id payload] (poll)
             tasks #{}]
        (chlog/handle-event event tasks id)
        (case event
          :queued-message-tasks (if (empty? tasks)
                                  true
                                  (recur (poll) tasks))

          :queued               (recur (poll) (conj tasks id))

          :started              (recur (poll) tasks)

          :finished             (let [new-tasks (disj tasks id)]
                                  (if (empty? new-tasks)
                                    true
                                    (recur (poll) new-tasks)))

          :exception            (throw payload)

          (if (seq tasks)
            false
            (recur (poll) tasks)))))))

(defn- wrapv
  "Turns `x` into `[x]`, unless it is sequential."
  [x]
  (cond-> x
    (not (sequential? x)) (vector)))

(defn- subscribe* [channel topic collection subscriber]
  (when (and topic subscriber)
    (let [{:keys [group subscribers]} channel
          factory (if (= group/REQUEST_RESPONSE group)
                    subscriber/->RequestResponse
                    subscriber/->Broadcast)]
      (log/debug topic collection "new" (name group) "subscriber")
      (doseq [t (wrapv topic)
              c (wrapv collection)]
        (dosync
          (alter subscribers
                 update-in
                 [group t c]
                 conj
                 (factory c subscriber)))))))

(defn- unsubscribe-all* [channel]
  (dosync
    (ref-set (:subscribers channel) {})))

(defn- route-message [message]
  (when-let [t (:task message)]
    (task/process t)))

(defn- broker-loop
  "Builds an infinitely looping function which:
    * takes messages out of `channel`
    * processes them
    * records state of `channel` in Prometheus
   Will block if `channel` is empty.
   Will wait for `throttle-interval`, if set."
  [channel & [throttle-interval]]
  (fn []
    (log/info "starting message broker")
    (while true
      (try
        (let [^BlockingQueue queue (:queue channel)
              message (.take queue)
              result (route-message message)]
          (chlog/on-message-processed channel message result))
        (when (and throttle-interval (not env/test?))
          (Thread/sleep throttle-interval))
        (catch Exception e
          (log/warn e "Exception in channel broker")
          (Thread/sleep 100))))))

(defprotocol Channel
  "Abstraction for media which transfer messages to subscribers.
   Subscription is done by declaring interest in a topic.
   Topics get published on the "
  (initialize [this])
  (publish [this context topic collection messages]
           "Puts `messages` on the channel.")
  (subscribe [this topic collection subscriber]
             "Registers `subscriber` for `topic` on `collection`.")
  (unsubscribe-all [this]
                   "Unregisters all subscribers."))

(defrecord RequestResponse [^BlockingQueue queue queue-id num_workers group subscribers]
  Channel

  (initialize [this]
    (dotimes [i num_workers]
      (doto (Thread. (broker-loop this) (str "mchan-rr-" i))
        (.setDaemon true)
        (.start)))
    this)
  
  (publish [this context topic collection messages]
    (publish* this context topic collection messages))

  (subscribe [this topic collection subscriber]
    (subscribe* this topic collection subscriber))

  (unsubscribe-all [this]
    (unsubscribe-all* this)))

(defrecord Broadcast [^BlockingQueue queue queue-id num_workers group subscribers]
  Channel

  (initialize [this]
    (dotimes [i num_workers]
      (doto (Thread. (broker-loop this BROADCAST_THROTTLE_INTERVAL_MS)
                     (str "mchan-bc-" i))
        (.setDaemon true)
        (.setPriority Thread/MIN_PRIORITY)
        (.start)))
    this)

  (publish [this context topic collection messages]
    (publish* this context topic collection messages))

  (subscribe [this topic collection subscriber]
    (subscribe* this topic collection subscriber))

  (unsubscribe-all [this]
    (unsubscribe-all* this)))

(defn create [group]
  (let [[f num_workers] (condp = group
                          group/REQUEST_RESPONSE [->RequestResponse
                                                  NUM_REQUEST_RESPONSE_WORKERS]
                          group/BROADCAST [->Broadcast
                                           NUM_BROADCAST_WORKERS])]
    (-> (PriorityBlockingQueue.)
        (f (str (UUID/randomUUID)) num_workers group (ref {}))
        (initialize))))
