(ns com.timezynk.useful.channel.logging
  (:require [clojure.tools.logging :as log]
            [com.timezynk.useful.channel.subscriber.hook :as hook]
            [com.timezynk.useful.prometheus.core :as prometheus])
  (:import [java.util.concurrent BlockingQueue]))

(def queue-size
  (prometheus/gauge :channel_queue_size
                    "Number of actions waiting in the channel queue"
                    :queue_id))

(def processed-messages
  (prometheus/counter :channel_processed_total
                      "Number of actions processed by the channel queue"
                      :queue_id))

(def time-in-queue
  (prometheus/gauge :channel_time_in_queue
                    "Time waiting in the channel queue (ms)"
                    :queue_id))

(def time-executing
  (prometheus/gauge :channel_time_executing
                    "Time executing the task (ms)"
                    :queue_id :collection :action :hook))

(def timed-out-messages
  (prometheus/counter :channel_timed_out_total
                      "Number of actions exceeding the wait-time threshold"))

(def ^:const ^:private WAIT_THRESHOLD
  "Maximum number of milliseconds to spend in queue before emitting a warning."
  10000)

(def ^:const ^:private RUN_THRESHOLD
  "Maximum number of milliseconds to spend running before emitting a warning."
  1000)

(defmulti handle-event
  (fn [event tasks id]
    (log/trace "E/ID/Q:" event "/" id "/" tasks)
    event))

(defmethod handle-event :queued-message-tasks
  [_ tasks _]
  (when (empty? tasks)
    (log/trace "completed - no tasks to wait for")))

(defmethod handle-event :queued
  [_ _ _]
  nil)

(defmethod handle-event :started
  [_ _ _]
  nil)

(defmethod handle-event :finished
  [_ tasks id]
  (when (empty? (disj tasks id))
    (log/trace "completed - all tasks finished")))

(defmethod handle-event :exception
  [_ _ _]
  nil)

(defmethod handle-event :default
  [_ tasks _]
  (when (seq tasks)
    (prometheus/inc! timed-out-messages)
    (log/info "timeout")))

(defn- log-wait-time [message queue-id millis]
  (when millis
    (-> time-in-queue (prometheus/gauge-with-labels queue-id) (.set millis))
    (log/logp (if (> millis WAIT_THRESHOLD) :warn :debug)
              (format "%s, W:%d" message millis))))

(defn- log-run-time [message queue-id millis]
  (when millis
    (let [{:keys [topic cname subscriber]} (:task message)
          ->label (comp name keyword)]
      (-> time-executing
          (prometheus/gauge-with-labels queue-id
                                        (->label cname)
                                        (->label topic)
                                        (hook/pretty-print (:f subscriber)))
          (.set millis))
      (log/logp (if (> millis RUN_THRESHOLD) :warn :debug)
                (format "%s, R:%d" message millis)))))

(defn on-message-processed
  "Records message stats in Prometheus."
  [channel message result]
  (let [{:keys [^BlockingQueue queue queue-id]} channel
        size-gauge (prometheus/gauge-with-labels queue-size queue-id)
        message-counter (prometheus/counter-with-labels processed-messages queue-id)
        [start-at end-at] result
        wait-time (and start-at (- start-at (:enqueued-at message)))
        run-time (and start-at end-at (- end-at start-at))]
    (.inc message-counter)
    (.set size-gauge (.size queue))
    (log-wait-time message queue-id wait-time)
    (log-run-time message queue-id run-time)))
