(ns com.timezynk.useful.mongo.queue
  (:require
   [clojure.tools.logging :as log]
   [com.timezynk.useful.prometheus.core :as metrics]
   [com.timezynk.useful.time :as t]
   [com.timezynk.useful.timed-queue :refer [TimedQueue]]
   [somnium.congomongo :as mongo]))

(def ^:const DEFAULT_NUM_WORKERS 1)
(def ^:const DEFAULT_MIN_INTERVAL 250)
(def ^:const DEFAULT_MIN_SLEEP 100)

(def ^:private ^:const JOIN_TIMEOUT
  "Maximum number of milliseconds to wait for worker threads to finish."
  20000)

(defprotocol WorkQueue
  (start-workers! [this thread-name] "Starts all worker threads.")
  (stop-workers! [this] "Stops all worker threads."))

(defn- pop-item! [queue]
  (mongo/fetch-and-modify (.collection queue)
                          {:run-at {:$lte (System/currentTimeMillis)}}
                          nil
                          :remove? true
                          :sort {:run-at 1}))

(defmacro ^:private report
  "Wraps `handle-form` such as to log the event and record statistics."
  [queue item handle-form]
  `(let [collection# (.collection ~queue)
         start-time# (System/nanoTime)
         item-type# (:type ~item)]
     (log/debug "Queue item of type" item-type# "in" collection# "is ready to run."
                "Latency" (- (System/currentTimeMillis) (:run-at ~item)) "ms")
     ~handle-form
     (metrics/inc! (.message-counter ~queue) item-type#)
     (metrics/inc-by! (.user-time ~queue)
                      (/ (double (- (System/nanoTime) start-time#))
                         1000000000.0)
                      item-type#)))

(defn- process-item! [queue item]
  (when item
    (let [handler (.handler queue)]
      (->> item (handler) (report queue item)))))

(defn- add-payload-prefix [acc [k v]]
  (assoc acc (str "payload." (name k)) v))

(defn- processing-loop
  "Processes `queue` until the termination condition is met."
  [queue]
  (let [collection (.collection queue)
        go-on? (.go-on? queue)
        min-interval (.min-interval queue)
        min-sleep (.min-sleep queue)]
    (log/info "Starting workers for queue" collection)
    (while @go-on?
      (try
        (->> (pop-item! queue)
             (process-item! queue)
             (t/sleep-pad min-interval min-sleep))
        (catch Exception e
          (log/error e "Exception in queue" collection))))
    (log/info "Queue" collection "finished")))

(deftype MongoQueue
         [collection handler num-workers thread-priority min-interval
          min-sleep go-on? worker-threads user-time message-counter]

  WorkQueue
  (start-workers! [this thread-name]
    (reset! go-on? true)
    (dotimes [i num-workers]
      (swap! worker-threads
             conj
             (doto (Thread. (bound-fn [] (processing-loop this))
                            (if (= 1 num-workers)
                              thread-name
                              (str thread-name "-" i)))
               (.setDaemon false)
               (.setPriority thread-priority)
               (.start)))))

  (stop-workers! [_this]
    (reset! go-on? false)
    (when (seq @worker-threads)
      (run! #(.join % JOIN_TIMEOUT) @worker-threads)
      (reset! worker-threads [])
      (metrics/unregister user-time)
      (metrics/unregister message-counter)))

  TimedQueue
  (push-job! [_this type run-at payload]
    (mongo/insert! collection
                   {:run-at run-at
                    :type type
                    :payload payload}
                   :write-concern :unacknowledged))

  (upsert-job! [_this type run-at selector update]
    (mongo/update! collection
                   (merge
                    {:type type}
                    (reduce add-payload-prefix {} selector))
                   (merge
                    {:$set
                     {:run-at run-at
                      :type type}}
                    update)
                   :upsert true
                   :write-concern :unacknowledged)))

(defn create [& {:as params}]
  (let [{:keys [id collection handler num-workers
                thread-priority min-interval min-sleep]} params]
    (MongoQueue. collection
                 handler
                 (or num-workers DEFAULT_NUM_WORKERS)
                 (or thread-priority Thread/MIN_PRIORITY)
                 (or min-interval DEFAULT_MIN_INTERVAL)
                 (or min-sleep DEFAULT_MIN_SLEEP)
                 (atom true)
                 (atom [])
                 (metrics/counter (keyword (str (name id) "_user_time_seconds"))
                                  (str "Total user time of " id)
                                  :type)
                 (metrics/counter (keyword (str (name id) "_messages_total"))
                                  (str "Messages processed in " id)
                                  :type))))
