(ns com.timezynk.useful.httpbus
  (:require [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [hato.websocket :as ws]
            [com.timezynk.useful.env :as env]
            [com.timezynk.useful.token :as token]))

(def ^:private HOST
  (if env/production?
    "api.timezynk.com"
    "nginx:8989"))

(def ^:private SCHEME
  (if env/production?
    "wss"
    "ws"))

(def ^:private PATH
  "/api/httpbus/v1/ws")

(defn- url
  "httpbus URL with query-string authorization."
  [company-id role]
  (->> (token/claims :company-id company-id :role role)
       (token/create)
       (str SCHEME "://" HOST PATH "?access_token=")))

(defn- log-error [error]
  (log/error error "Sending to httpbus failed"))

(defn send!
  "Sends `payload` to the httpbus WebSocket.
   Authorizes on the basis of `company-id` and `(:role options)`."
  [payload company-id & {:as options}]
  (let [{:keys [role] :or {role :owner}} options
        socket @(ws/websocket (url company-id role)
                              {:on-error log-error})]
    (try
      (->> payload
           (json/encode)
           (ws/send! socket))
      (catch Exception e
        (log/error e "Failed to publish to httpbus"))
      (finally
        (ws/close! socket)))
    payload))

(defn create!
  "Publishes a `create` event for the specified documents."
  [domain-type ids company-id & {:as options}]
  (send! [{:type domain-type
           :action :create
           :ids ids}]
         company-id
         options))

(defn update!
  "Publishes an `update` event for the specified documents."
  [domain-type ids company-id & {:as options}]
  (send! [{:type domain-type
           :action :update
           :ids ids}]
         company-id
         options))
