(ns com.timezynk.useful.jwt
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clj-jwt.core :as jwt]
            [clj-jwt.key :as jwt-key]))

(def ^:private ^:const ISS
  (or (System/getenv "JWT_ISS") "https://pro.timezynk.com/session"))

(def ^:private ^:const AUD
  (or (System/getenv "JWT_AUD") "https://pro.timezynk.com"))

(def ^:private ^:const PUBLIC_KEY_FILENAME
  (or (System/getenv "PUBLIC_KEY_FILENAME") "public_key.pem"))

(def ^:private rsa-pub-key
  (some-> PUBLIC_KEY_FILENAME io/resource jwt-key/public-key))

(defn decode
  "Attempts to decode `s`. Swallows all exceptions, returns `nil` on failure."
  [s]
  (when s
    (try
      (jwt/str->jwt s)
      (catch Exception e
        (log/warn e "Exception 401: Failed to decode JWT" s)
        nil))))

(defn not-expired? [jwt timestamp]
  (let [claims (:claims jwt)
        expires (* (:exp claims 0) 1000)]
    (when (and (< timestamp expires)
               (= ISS (:iss claims))
               (= AUD (:aud claims)))
      jwt)))

(defn signed? [jwt]
  (if (jwt/verify jwt :RS256 rsa-pub-key)
    jwt
    (log/info "Exception 401: Invalid JWT signature for sub"
              (get-in jwt [:claims :sub]))))

(defn verify
  "`token` if valid, `nil` otherwise."
  ([token]
   (verify token (System/currentTimeMillis)))
  ([token timestamp]
   (let [jwt (cond-> token
               (string? token) (decode))]
     (when (and jwt
                (signed? jwt)
                (not-expired? jwt timestamp))
       token))))
