(ns com.timezynk.useful.quartzite
  (:require [clojure.string :as string]
            [clojure.tools.logging :as log]
            [clojurewerkz.quartzite.jobs :as j]
            [clojurewerkz.quartzite.scheduler :as qs]
            [clojurewerkz.quartzite.triggers :as t]
            [com.timezynk.useful.quartzite.cron :as cron]
            [com.timezynk.useful.quartzite.context :as context]))

(defonce ^:private scheduler (atom nil))

(defn schedule
  "Schedules a job to a CRON specification.
   Duplicates the thread bindings of the caller for the scheduled job."
  ([scheduler cron-expr job-class]
   (let [job (j/build
              (j/of-type job-class)
              (j/using-job-data (context/dehydrate (get-thread-bindings)))
              (j/with-identity (j/key (str "jobs." (.getName job-class)))))
         trigger (t/build
                  (t/with-identity (t/key (str "triggers." (rand-int 1000000))))
                  (t/start-now)
                  (t/with-schedule (cron/schedule cron-expr)))]
     (qs/schedule scheduler job trigger)))
  ([cron-expr job-class]
   (schedule @scheduler cron-expr job-class)))

(def ^:private job-class-ok?
  (every-pred symbol?
              (comp nil? resolve)))

(defmacro defjob
  "Generates a Quartz job definition.
   The definition includes a catch-all clause which logs at level `ERROR`."
  [job-class & body]
  {:clj-kondo/ignore [:unresolved-symbol]}
  {:pre [(job-class-ok? job-class)]}
  `(j/defjob ~job-class
     [~'context]
     (with-bindings (context/hydrate ~'context)
       (let [unqualified# (-> ~job-class (str) (string/split #"\.") (peek))]
         (try
           (log/info (format "Finished scheduled job: %s (%s ms)"
                             unqualified#
                             (->> (do ~@body)
                                  (time)
                                  (with-out-str)
                                  (re-find #"(\d+)\.\d+ msecs")
                                  (second))))
           (catch Exception e#
             (log/error e# "Failed to run background job" unqualified#)))))))

(defn start
  "Starts a Quartz scheduler and returns it."
  []
  (->> (qs/initialize)
       (qs/start)
       (reset! scheduler)))

 (defn stop
   "Stops `scheduler`. If none given, stop the most recently started scheduler."
   ([scheduler]
    (qs/shutdown scheduler))
   ([]
    (when @scheduler
      (qs/shutdown @scheduler)
      (reset! scheduler nil))))
