(ns com.timezynk.useful.mongo
  (:require
   [clojure.core.reducers :as r]
   [clojure.set :as cs]
   [potemkin :as pk]
   [slingshot.slingshot :refer [throw+]]
   [validateur.validation :as validation])
  (:import [org.bson.types ObjectId]))

(def ^:const OBJECT_ID_REGEX #"^[\da-f]{24}$")

(defprotocol PotentialObjectId
  (object-id [o] "Convert to object id"))

(extend-protocol PotentialObjectId
  ObjectId
  (object-id [o] o)
  clojure.lang.Named
  (object-id [o] (ObjectId. (name o)))
  Object
  (object-id [o] (ObjectId. (str o)))
  nil
  (object-id [_o] nil))

(defn object-ids [ids]
  (when ids
    (map object-id ids)))

(defn object-id? [s]
  (or
   (instance? ObjectId s)
   (and (string? s) (re-matches OBJECT_ID_REGEX s))))

(defn unpack-id [entry]
  (cs/rename-keys entry {:_id :id}))

(defn pack-id [entry]
  (cs/rename-keys entry {:id :_id}))

(defn unpack-ids [entries]
  (map unpack-id entries))

(defn copy-attr
  ([e k] (copy-attr e k object-id))
  ([e k f]
   (when (contains? e k)
     {k (f (get e k))})))

(defmacro map-copy-attr
  ([e ks]
   `(apply merge (map (fn [k#] (copy-attr ~e k#)) ~ks)))
  ([e ks f]
   `(apply merge (map (fn [k#] (copy-attr ~e k# ~f)) ~ks))))

(defn object-ids-q [k ids]
  (when ids
    {k {:$in (object-ids ids)}}))

(defn object-id-q [k id]
  (when id
    {k (object-id id)}))

(defprotocol DoubleConvertible
  (to-double [d] "Convert to double"))

(extend-protocol DoubleConvertible
  String
  (to-double [s]
    (try
      (Double/parseDouble s)
      (catch NumberFormatException _e
        nil)))
  Double
  (to-double [d] d)
  Integer
  (to-double [i] (.doubleValue i))
  nil
  (to-double [_o] nil))

(defprotocol ConvertibleToLong
  (to-long [o] "Convert to long"))

(extend-protocol ConvertibleToLong
  String
  (to-long [s] (Long/parseLong s))
  Long
  (to-long [l] l)
  Integer
  (to-long [i] (.longValue i)))

(defn nil-empty-strings [a]
  (r/reduce
   (fn [a k v]
     (if (and (string? v) (.isEmpty ^String v))
       (assoc a k nil)
       a))
   a a))

(defn get-id-in [m ks]
  (object-id (get-in m ks)))

(defn get-ids-in [m ks]
  (object-ids (get-in m ks)))

(defn extend-defaults [defaults entry]
  (merge defaults (select-keys entry (keys defaults))))

(defn validate-string [& strings]
  (every? #(and % (string? %) (seq %)) strings))

(defn validate-object-id [& ids]
  (every?
   #(try
      (object-id %)
      (catch Exception _e
        nil)) ids))

(defn object-id-or-self [id]
  (try
    (object-id id)
    (catch Exception _ id)))

(defmacro defvalidator [fn-name & body]
  `(def ~fn-name
     (let [vs# (validation/validation-set ~@body)]
       (fn [entry#]
         (let [errors# (vs# entry#)]
           (if (empty? errors#)
             entry#
             (throw+ {:code 400 :error :validation :message errors#})))))))

(defn validate-all [vs entries]
  (r/reduce
   (fn [result e]
     (merge-with cs/union result (vs e)))
   {} entries))

(pk/import-fn validation/exclusion-of)
(pk/import-fn validation/format-of)
(pk/import-fn validation/inclusion-of)
(pk/import-fn validation/length-of)
(pk/import-fn validation/numericality-of)
(pk/import-fn validation/presence-of)
(pk/import-fn validation/acceptance-of)

(defn object-id-of [attribute]
  (fn [e]
    (let [v (get e attribute)]
      (if (and v (validate-object-id v))
        [true {}]
        [false {attribute #{(str "Invalid ObjectId in " attribute)}}]))))

(defmacro child-of [attribute & body]
  `(let [a# ~attribute
         vs# (validation/validation-set ~@body)]
     (fn [e#]
       (let [errors# (vs# (get e# a#))]
         (if (empty? errors#)
           [true {}]
           [false {a# #{errors#}}])))))

(defmacro children-of [attribute & body]
  `(let [a# ~attribute
         vs# (validation/validation-set ~@body)]
     (fn [e#]
       (let [errors# (validate-all vs# (get e# a#))]
         (if (empty? errors#)
           [true {}]
           [false {a# #{errors#}}])))))
