(ns com.timezynk.useful.session.middleware
  (:require [clojure.tools.logging :as log]
            [com.timezynk.useful.jwt :as jwt]
            [com.timezynk.useful.mongo :as um]
            [com.timezynk.useful.rest.current-user :as current]))

(def ^:private ^:const HEADER_PATTERN #"^Bearer (\S+)$")

(defn- decode-header
  "Converts the HTTP header to a JWT token."
  [header]
  (when header
    (-> (re-matches HEADER_PATTERN header)
        (get 1)
        (jwt/decode))))

(defn- decode-request
  "Converts the HTTP request to a JWT token."
  [request]
  (let [{:keys [headers params]} request]
    (or
     (decode-header (or (get headers "authorization")
                        (get headers "x-tz-authorization")))
     (jwt/decode (get params :access_token)))))

(defn- decode-token
  "Converts the JWT token to a JWT claims hashmap."
  [token]
  (when (jwt/verify token)
    (:claims token)))

(defn- ->session [request inflate-fn]
  (try
    (-> request decode-request decode-token inflate-fn)
    (catch Exception e
      (log/warn e "Exception 401: Exception while building session")
      nil)))

(defn parse-claims
  "Converts the JWT claims to a session hashmap."
  [claims]
  (when claims
    (let [{:keys [jti sub x-tz-rol x-tz-cid x-tz-iid]} claims]
      {:id (um/object-id jti)
       :user {:id (um/object-id sub)
              :role (keyword x-tz-rol)
              :company-id (um/object-id x-tz-cid)
              :identity-id (um/object-id x-tz-iid)}
       :claims claims})))

(defn wrap-session
  "Decorates requests with the results of
   `com.timezynk.useful.session.middleware/parse-claims`
   under the `:session` key."
  ([handler]
   (wrap-session handler parse-claims))
  ([handler inflate-fn]
   (fn [request]
     (-> request
         (assoc :session (->session request inflate-fn))
         (handler)
         (dissoc :session)))))

(defn wrap-current
  "Binds `com.timezynk.useful.rest.current-user` helpers
   to the authenticated user."
  [handler]
  (fn [request]
    (current/with-user (get-in request [:session :user])
      (handler request))))
