(ns com.timezynk.useful.rest.middleware
  (:require [slingshot.slingshot :refer [try+]])
  (:require [clojure.spec.alpha :refer [check-asserts check-asserts?]]
            [clojure.tools.logging :as log]
            [ring.util.response :as res]
            [com.timezynk.useful.string :as s]
            [com.timezynk.useful.rest.middleware.handler :as handler]
            [com.timezynk.useful.rest.middleware.logger :as logger]))

(def ^:const SLOW_LOG_LIMIT 0.5)

(def rng (java.util.Random. (System/currentTimeMillis)))

(defn wrap-exceptions [handler]
  (fn [request]
    (try+
     (when-not (check-asserts?)
       (check-asserts true))
     (handler request)
     (catch [:clojure.spec.alpha/failure :assertion-failed] e
       (logger/spec request (:throwable &throw-context))
       (handler/spec e))
     (catch [:type :validation-error] e
       (logger/validation request (:throwable &throw-context))
       (handler/validation e))
     (catch [:error :cancan] e
       (logger/authorization request &throw-context)
       (handler/authorization e))
     (catch [:code 401] e
       (logger/authorization request &throw-context)
       (handler/authorization e))
     (catch map? e
       (logger/slingshot request &throw-context)
       (handler/slingshot e))
     (catch Exception e
       (logger/generic request e)
       (handler/generic e)))))

(defn wrap-nocache [handler]
  (fn [request]
    (when-let [result (handler request)]
      (cond
        (= :post (:request-method request)) (res/header result "Cache-Control" "no-cache")
        (.endsWith ^String (:uri request) ".js") (res/header result "Cache-Control" "max-age=1,must-revalidate,public")
        :else result))))

(defn wrap-cors [handler]
  (fn [request]
    (if (= :options (:request-method request))
      (-> {:status 200
           :body ""}
          (res/header "Access-Control-Allow-Origin" "*")
          (res/header "Access-Control-Max-Age" "86400")
          (res/header "Access-Control-Allow-Headers" "content-type, accept, origin, authorization, x-tz-authorization, x-tz-client, user-agent, accept-encoding")
          (res/header "Access-Control-Allow-Methods" "GET, POST, PUT, PATCH, DELETE, OPTIONS"))
      (when-let [result (handler request)]
        (-> result
            (res/header "Access-Control-Allow-Origin" "*"))))))

(defn wrap-performance-alert [handler]
  (fn [request]
    (let [start-time (System/currentTimeMillis)
          response (handler request)
          finish-time (System/currentTimeMillis)
          request-time (-> (- finish-time start-time) (double) (/ 1000.0))]
      (when (> request-time SLOW_LOG_LIMIT)
        (log/warn "Slow request" request-time "seconds:" (s/from-request request)))
      response)))
