(ns com.vadelabs.adapter-xtdb.main
  (:require
   [clojure.walk :as cwalk]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.string :as ustr]
   [com.wsscode.pathom3.connect.operation :as pco]
   [edn-query-language.core :as eql]
   [xtdb.api :as xt]))

(def transaction-functions
  {::delta-update
   '(fn [ctx [id {:keys [before after]}]]
      (let [db (xtdb.api/db ctx)
            entity (xtdb.api/entity db id)
            before (when entity
                     (reduce (fn [e [k v]]
                               (cond-> e
                                 (and (some? v) (get e k)) (assoc k v)))
                       entity
                       before))]
        [[:xtdb.api/match id before]
         [:xtdb.api/put (merge entity after)]]))})

(defn ref?
  [attributes-map k]
  (= :attribute.type/ref (some-> k attributes-map :attribute/local-type)))

(defn ->delta-update-txs
  [doc-delta]
  (reduce
    (fn [acc [id delta]]
      (conj acc [::xt/fn ::delta-update [id delta]]))
    []
    doc-delta))

(defn is-ident?
  [value]
  (and (vector? value)
    (= 2 (count value))
    (keyword? (first value))
    (or (uuid? (second value))
      (uc/tempid? (second value)))))

(defn ident->id
  [[_attr key] tempid->generated-id]
  key)

(defn idents->ids
  [doc tempid->generated-id {:keys [attributes-map]}]
  (reduce
    (fn [acc [k v]]
      (assoc acc k (cond
                     (is-ident? v) (second v)
                     (and (vector? v) (every? is-ident? v)) (mapv second v)
                     :else v)))
    {}
    doc))

(defn ->doc-delta
  "Converts attribute delta into document delta"
  [attr-delta tempid->generated-id aenv]
  (let [resp (reduce
               (fn [acc [[attr key] doc]]
                 (let [key (get tempid->generated-id key key)]
                   (assoc-in acc [key]  (-> (reduce
                                              (fn [acc [attr {:keys [before after] :as value}]]
                                                (let [after (or after value)]
                                                  (-> acc
                                                    (assoc-in [:before attr] before)
                                                    (assoc-in [:after attr] after))))
                                              {}
                                              doc)
                                          (assoc-in [:after attr] key)
                                          (assoc-in [:after :xt/id] key)
                                          (update :before idents->ids tempid->generated-id aenv)
                                          (update :after idents->ids tempid->generated-id aenv)))))
               {}
               attr-delta)]
    resp))

(defn tempids->generated-ids
  [delta]
  (->> (keys delta)
    (keep (fn [[k id]]
            (let [suggestion (keyword (if (seq (:id id))
                                        (->> (:id id)
                                          (map uc/stringify)
                                          (ustr/join "_"))
                                        (:id id)))]
              (when (uc/tempid? id)
                [id (uc/uuid suggestion)]
                [id id]))))
    (into {})))

(defn delta->txn
  [aenv delta]
  (let [tempid->generated-id (tempids->generated-ids delta)]
    {:tempid->generated-id tempid->generated-id
     :txn (-> delta
            (->doc-delta tempid->generated-id aenv)
            ->delta-update-txs)}))

(defn save!
  [{:keys [attributes-map node] :as aenv} action penv {:keys [data]}]
  (let [delta (uc/normalize data)
        {:keys [txn]} (delta->txn aenv delta)
        tx (xt/submit-tx node txn)]
    (xt/await-tx node tx)
    (when-not (xt/tx-committed? node tx)
      (println "Transaction failed!"))
    (mapv (fn [[attr-key temp-id]]
            (let [qkey (get-in attributes-map [attr-key :attribute/qualified-key])]
              {qkey temp-id}))
      (keys delta))))

(defn delete!
  [{:keys [node]} action penv pparams]
  (let [pk (ffirst pparams)
        eid (get pparams pk)
        tx (xt/submit-tx node [[::xt/delete eid]])]
    (xt/await-tx node tx)))

(defn resolve-aliased-attrs
  [{:keys [attributes-map]} output]
  (cwalk/prewalk
    (fn [node]
      (if (qualified-keyword? node)
        (get-in attributes-map [node :attribute/remote-key])
        node))
    output))

(defn ^:private local-result
  [{:keys [attributes-map]} ast-nodes remote-result]
  (if (seq ast-nodes)
    (reduce
      (fn [acc {:keys [dispatch-key type]}]
        (let [{:attribute/keys [remote-key target]} (get attributes-map dispatch-key)
              result-value (or (get remote-result remote-key)
                             (get remote-result dispatch-key)) ;; Fallback to dispatch-key if result is nil from remote-key
              #_local-type #_(get-in attributes-map [target :attribute/local-type] local-type)
              #_result-value #_(uc/coerce local-type result-value)]
          (cond-> acc
            (= type :join) (assoc-in [dispatch-key target] result-value)
            (= type :prop) (assoc-in [dispatch-key] result-value))))
      {}
      ast-nodes)
    remote-result))

(defn ^:private local-results
  [adapter ast-nodes remote-results]
  (if (map? remote-results)
    (local-result adapter ast-nodes remote-results)
    (mapv (partial local-result adapter ast-nodes) remote-results)))

(defn ^:private where-clause
  [{:keys [attributes-map]} pinput]
  (if (seq pinput)
    (reduce-kv
      (fn [acc key value]
        (let [entity-key (get-in attributes-map [key :attribute/remote-key])]
          (conj acc ['?entity entity-key (or value '_)])))
      []
      pinput)
    [['?entity :xt/id '_]]))

(defn remote-query
  [aenv penv pinput desired-output]
  (let [desired-output (if (seq desired-output) desired-output ['*])
        {:keys [limit offset]} (pco/params penv)]
    (cond-> {:find [`(~'pull ~'?entity ~desired-output)]
             :where (where-clause aenv pinput)}
      limit (assoc :limit limit)
      offset (assoc :offset offset))))

(defn execute-query
  ([{:keys [node attributes-map]} penv pinput desired-output]
   (let [db (xt/db node)
         desired-output (or desired-output ['*])
         where-clause (where-clause attributes-map pinput)
         {:keys [limit offset]} (pco/params penv)
         remote-query (cond-> {:find [`(~'pull ~'?entity ~desired-output)]
                               :where where-clause}
                        limit (assoc :limit limit)
                        offset (assoc :offset offset))
         results (xt/q db remote-query)]
     (->> results
       (map first)
       (filterv seq)))))

(defn exec-query
  [{:keys [node]} query]
  (let [results (-> node
                  xt/db
                  (xt/q query))]
    (->> results
      (map first)
      (filterv seq))))

(defn user-shape
  [penv]
  (get-in penv
    [:com.wsscode.pathom3.connect.planner/graph
     :com.wsscode.pathom3.connect.planner/user-request-shape]))

;; TODO: fix it
(defn eql-query
  [user-shape {:action/keys [identifier] :as action}]
  (reduce-kv (fn [acc k v]
               (if (or (= identifier k) (= :com.wsscode.pathom3.connect.runner/attribute-errors k))
                 acc ;; return acc if action is
                 (if (seq v)
                   (conj acc {k (eql-query v action)})
                   (conj acc k))))
    []
    user-shape))

(defn ^:private ->remote-key
  [attributes-map qualified-key]
  (or (get-in attributes-map [qualified-key :attribute/remote-key])
    (uc/raise :type :not-found
      :code :remote-key-not-found
      :qualified-key qualified-key
      :hint (ustr/format "couldn't find remote key for %s" qualified-key))))

(defn local-query
  [{:keys [attributes-map]} {:action/keys [identifier]} tree]
  (cwalk/prewalk
    (fn [node]
      (if (and (qualified-keyword? node) (not= node identifier))
        (->remote-key attributes-map node)
        node))
    tree))

(defn local-result
  [{:keys [attributes-map] :as aenv} ast-nodes result-map]
  (reduce
    (fn [acc {:keys [children dispatch-key type]}]
      (let [{:attribute/keys [remote-key target]} (get attributes-map dispatch-key)
            result-value (or (get result-map remote-key)
                           (get result-map dispatch-key))]
        (cond-> acc
          (= type :join) (assoc dispatch-key (if (map? result-value)
                                               (local-result aenv children result-value)
                                               (mapv (partial local-result aenv children) result-value)))
          (= type :prop) (assoc dispatch-key result-value))))
    {}
    ast-nodes))

(defn local-results
  [aenv aliased-query results]
  (let [{:keys [children]} (eql/query->ast aliased-query)]
    (if (map? results)
      (local-result aenv children results)
      (mapv (partial local-result aenv children) results))))

(defn query
  [aenv {:action/keys [type identifier output] :as action} penv pinput]
  (let [eql-query (-> penv user-shape (eql-query action))
        nspaced-query (if (seq eql-query) eql-query output)
        unaliased-query (local-query aenv action nspaced-query)
        remote-query (remote-query aenv penv pinput unaliased-query)
        remote-results (exec-query aenv remote-query)
        local-results (local-results aenv nspaced-query remote-results)]
    #_(tap> {:rr remote-results
             :lr local-results
             :nq nspaced-query
             :uq unaliased-query
             :rq remote-query})
    (case type
      :list {identifier local-results}
      :ident (or (first local-results) {})
      (throw (ex-info "Type not supported" {:action action})))))
