(ns com.vadelabs.adapter-xtdb.resolvers
  (:require
   [clojure.walk :as cwalk]
   [com.wsscode.pathom3.connect.operation :as pco]
   [edn-query-language.core :as eql]
   [xtdb.api :as xt]))

(defn eql-query->xtdb-query
  "Converts EQL query to XTDB query"
  [attributes-map pathom-query]
  (let [identity? #(true? (:attribute/identity? %))
        resolvers (into #{}
                    (comp
                      (filter identity?)
                      (mapcat (fn [{:attribute/keys [atlas-key remote-key]}]
                                [atlas-key remote-key])))
                    (vals attributes-map))]

    (cwalk/prewalk (fn [e]
                     (let [remote-key (get-in attributes-map [e :attribute/remote-key])]
                       (cond
                         (and (keyword? e) (contains? resolvers e)) :xt/id
                         remote-key remote-key
                         :else e)))
      pathom-query)))

(defn- fix-id-keys
  [attributes-map ast-nodes result]
  (let [join-key->children (into {}
                             (comp
                               (filter #(= :join (:type %)))
                               (map (fn [{:keys [key children]}] [key children])))
                             ast-nodes)
        join-keys          (set (keys join-key->children))
        join-key?          #(contains? join-keys %)]

    (reduce
      (fn [acc {:keys [dispatch-key]}]
        (let [attribute                                          (get attributes-map dispatch-key)
              {:attribute/keys [atlas-key remote-key identity?]} attribute
              remote-value                                       (if identity?
                                                                   (get result :xt/id)
                                                                   (get result remote-key))]
          (if remote-value
            (cond
              identity?
              (assoc acc atlas-key remote-value)

              (and (join-key? atlas-key) (vector? remote-value))
              (assoc acc atlas-key (mapv (partial fix-id-keys attributes-map (join-key->children atlas-key)) remote-value))

              (and (join-key? atlas-key) (map? remote-value))
              (assoc acc  atlas-key (fix-id-keys attributes-map (join-key->children atlas-key) remote-value))

              :else (assoc acc atlas-key remote-value))
            acc)))
      {}
      ast-nodes)))

(defn- fix-keys
  [attributes-map ast-nodes result]
  (let [join-key->children (into {}
                             (comp
                               (filter #(= :join (:type %)))
                               (map (fn [{:keys [key children]}] [key children])))
                             ast-nodes)
        join-keys          (set (keys join-key->children))
        join-key?          (partial contains? join-keys)]

    (reduce
      (fn [acc {:keys [dispatch-key]}]
        (let [attribute                                                             (get attributes-map dispatch-key)
              {:attribute/keys [atlas-key remote-key outputs component? identity?]} attribute
              {:keys [children]}                                                    (eql/query->ast outputs)
              remote-value                                                          (if identity?
                                                                                      (get result :xt/id)
                                                                                      (get result remote-key))]
          (cond
            (and (seq children) component? (vector? remote-value))
            (assoc acc atlas-key (mapv (partial fix-keys attributes-map children) remote-value))

            (and (seq children) component? (map? remote-value))
            (assoc acc atlas-key (fix-keys attributes-map children remote-value))

            (and (seq children) (vector? remote-value))
            (assoc acc atlas-key (mapv (partial fix-keys attributes-map children) remote-value))

            (and  (seq children) (vector? result))
            (assoc acc atlas-key (mapv (partial fix-keys attributes-map children) result))

            (and (join-key? atlas-key) (vector? remote-value))
            (assoc acc atlas-key (mapv (partial fix-keys attributes-map (join-key->children atlas-key)) remote-value))

            (and (join-key? atlas-key) (map? remote-value))
            (assoc acc atlas-key (fix-keys attributes-map (join-key->children atlas-key) remote-value))

            :else
            (assoc acc atlas-key remote-value))))

      {}
      ast-nodes)))

(defn- pathom-result
  [env {:adapter/keys [type id]} pathom-query result]
  (let [attributes-map     (get-in env [type id :attributes-map])
        {:keys [children]} (eql/query->ast pathom-query)]
    (if (vector? result)
      (mapv (partial fix-keys attributes-map children) result)
      (fix-keys attributes-map children result))))

(defn- where-clause
  [attributes-map params]
  (reduce-kv
    (fn [acc key value]
      (let [entity-key (get-in attributes-map [key :attribute/remote-key])]
        (conj acc ['?entity entity-key (or value '_)])))
    []
    params))

(defn- execute-query
  [env {:adapter/keys [type id]} input desired-output]
  (let [attributes-map (get-in env [type id :attributes-map])
        params         (pco/params env)
        limit          (:limit params 1000)
        offset         (:offset params 0)
        filter         (:filter params)
        input          (merge input filter)
        node           (get-in env [type id :node])
        db             (xt/db node)
        where-clause   (where-clause attributes-map input)
        results        (xt/q db
                         {:find   [`(~'pull ~'?entity ~desired-output)]
                          :where  where-clause
                          :limit  limit
                          :offset offset})]
    (mapv #(first %) results)))

(defn- pathom-resolve-fn
  [{:adapter/keys [type id]
    :as           adapter} {:attribute/keys [atlas-key target dependencies]} outputs]
  (fn [env input]
    (let [attributes-map (get-in env [type id :attributes-map])
          input          (if (empty? dependencies) (assoc input target nil) input)
          xtdb-query     (eql-query->xtdb-query attributes-map outputs)
          result         (execute-query env adapter input xtdb-query)
          final-result   (pathom-result env adapter outputs result)]
      (if target
        {atlas-key final-result}
        (first final-result)))))

(defn generate
  "Generate resolvers that make sense for the given set of attributes."
  [{:adapter/keys [type id]
    :as           adapter}]
  (let [attributes-grouped (get-in {} [type id :attributes-grouped])]
    (reduce-kv
      (fn [acc [op-name input] output-attributes]
        (let [{:attribute/keys [atlas-key]
               :as             attribute} (first output-attributes)
              eql-query                                    [] #_(atlas.core/attributes->eql output-attributes)
              op-name                                      (symbol (namespace op-name) (name op-name))
              pathom-config                                (cond-> {::pco/op-name op-name
                                                                    ::pco/resolve (pathom-resolve-fn adapter attribute eql-query)}
                                                             (empty? input) (assoc
                                                                              ::pco/output [{atlas-key eql-query}])
                                                             (seq input) (assoc
                                                                           ::pco/output eql-query
                                                                           ::pco/input input))]
          (conj acc (pco/resolver pathom-config))))
      []
      attributes-grouped)))
