(ns com.vadelabs.sql-core.diff
  (:require
   [com.vadelabs.utils-core.interface :as uc]))

(defn ^:private changes
  [prev-model next-model]
  (let [changes (uc/changes prev-model next-model)
        created (uc/created changes prev-model next-model)
        dropped (uc/dropped changes prev-model next-model)
        altered (uc/altered changes prev-model next-model)]
    {:created created :dropped dropped :altered altered}))

(defn ^:private create-enum
  [created _ next-model]
  (let [enums (if (vector? created) created [created])]
    (mapv
      (fn [enum]
        (into
          [:create-type {:as :enum} (get-in next-model [enum :name])]
          (get-in next-model [enum :options])))
      enums)))

(defn ^:private drop-enum
  [dropped prev-model next-model])

(defn ^:private alter-enum
  [altered prev-model next-model])

(defn enums
  [result-set {prev-model :enums} {next-model :enums}]
  (let [{:keys [created dropped altered] :as changes} (changes prev-model next-model)]
    (cond-> result-set
      (seq created) (into (create-enum created prev-model next-model))
      (seq dropped) (into (drop-enum dropped prev-model next-model))
      (seq altered) (into (alter-enum altered prev-model next-model)))))

(defn ^:private columns
  [cols]
  (->> cols
    vals
    (map (juxt :name :props :type))))

(defn ^:private primary-key
  [{:keys [name fields]}]
  (when (seq fields)
    (into [:primary-key {} name] fields)))

(defn ^:private create-table
  [created _ next-model]
  (let [tables (if (vector? created) created [created])]
    (mapv
      (fn [table]
        (let [columns (columns (get-in next-model [table :columns]))
              primary-key (primary-key (get-in next-model [table :primary-key]))
              table-name (get-in next-model [table :name])]
          (cond-> [:create-table {:if-not-exists true}
                   table-name]
            (seq columns) (into columns)
            (seq primary-key) (conj primary-key))))
      tables)))

(defn ^:private drop-table
  [dropped prev-model next-model])

(defn ^:private alter-table
  [altered prev-model next-model])

(defn tables
  [result-set {prev-model :tables} {next-model :tables}]
  (let [{:keys [created dropped altered]} (changes prev-model next-model)]
    (cond-> result-set
      (seq created) (into (create-table created prev-model next-model))
      (seq dropped) (into (drop-table dropped prev-model next-model))
      (seq altered) (into (alter-table altered prev-model next-model)))))
