(ns com.vadelabs.sql-core.query
  (:require
   [com.vadelabs.sql-core.relation :as relation]
   [com.vadelabs.sql-core.params :as params]
   [com.vadelabs.sql-core.schema :as schema]
   [com.vadelabs.sql-core.result-set :as result-set]
   [com.vadelabs.sql-core.coerce :as c]
   [com.vadelabs.sql-core.aenv :as env]
   [com.vadelabs.sql-core.tree :as tree]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]))

(defn into-query
  "Transform the output of `params/for-query` into a valid
   honeysql query."
  [schema {:keys [table selections conditions joins]}]
  (cond-> {:select selections
           :from table}

    (seq conditions)
    (assoc :where (if (= 1 (count conditions))
                    (first conditions)
                    (concat [:and] conditions)))

    (seq joins)
    (assoc :left-join
      (mapcat (partial relation/resolve-and-expand schema) joins))))

(defn sql-query
  "Return an SQL query ready for `next.jdbc/execute` based on the
  output of `params/for-query`."
  [schema params]
  (->> params
    (into-query schema)
    hsql/format))

(defn educt
  "Run a EQL query and return an eduction of the records.
   No tree recomposition is performed, joined rows will be
   returned one by one. Since EQL's tree recomposition sorts
   and groups records it must hold all of the returned result
   set in memory. When querying large datasets, `educt` can be
   used to limit memory usage and ensure faster consumption."
  ([aenv entity fields conditions]
   (let [params (params/for-query (env/schema aenv) entity fields conditions)]
     (educt aenv params)))
  ([{:keys [schema jdbc]} params]
   (let [opts {:builder-fn (result-set/builder-fn schema)}
         q (sql-query schema params)]
     (eduction (map c/read-map) (jdbc/execute! jdbc q opts)))))

(defn execute
  "Execute an EQL query. Accepts adapter environment and
   the target entity or an ident specification. Optionaly
   takes the list of fields (including relation fields) to
   resolve and conditions."
  ([aenv entity]
   (execute aenv entity (schema/resolve-fields (env/schema aenv) entity) []))
  ([aenv entity fields]
   (execute aenv entity fields []))
  ([aenv entity fields conditions]
   (let [schema (env/schema aenv)
         {:keys [joins ident?] :as params} (params/for-query schema entity fields conditions)
         results (educt aenv params)]
     (cond
       (and (seq joins) ident?)
       (first (tree/build schema params results))

       ident?
       (first results)

       (seq joins)
       (tree/build schema params results)

       :else
       (into [] results)))))
